#include <stdio.h>
#include "coarray.h"

namespace COLIB {
    static void read_pnm_header(FILE *stream,char &ptype,int &w,int &h,int &maxval) {
	char c;
	if(fscanf(stream,"%c%c",&c,&ptype)!=2) throw "read short";
	if(c!='P') throw "not a pnm file";
	while(fscanf(stream," #%*[^\n]%c",&c)==1);
	if(fscanf(stream,"%d",&w)!=1) throw "format error (width)";
	if(ptype=='1' || ptype=='4') {
	    maxval = 1;
	    while(fscanf(stream," #%*[^\n]%c",&c)==1);
	    if(fscanf(stream,"%d%c",&h,&c)!=2) throw "format error (height)";
	} else {
	    while(fscanf(stream," #%*[^\n]%c",&c)==1);
	    if(fscanf(stream,"%d",&h)!=1) throw "format error (height)";
	    while(fscanf(stream," #%*[^\n]%c",&c)==1);
	    if(fscanf(stream,"%d%c",&maxval,&c)!=2) throw "format error (maxval)";
	}
	if(c!=' ' && c!='\n' && c!='\r' && c!='#') throw "format error (header end)";
    }

    inline int safe_getc(FILE *stream) {
	int c = getc(stream);
	if(c==EOF) throw "read error";
	return c;
    }

    inline int safe_putc(char c,FILE *stream) {
	int result = putc(c,stream);
	if(result==EOF) throw "write error";
	return result;
    }

    void read_pnm(FILE *stream,array2<unsigned char> &image) {
	char ptype;
	int w,h,maxval;
	read_pnm_header(stream,ptype,w,h,maxval);
	if(maxval<0||maxval>255) throw "cannot handle 16bpp PNM files yet";
	if(ptype=='1') {
	    image.resize(w,h);
	    for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
		int c;
		for(;;) {
		    c = safe_getc(stream);
		    if(c==' '||c=='\r'||c=='\t'||c=='\n') continue;
		    if(c=='0') {image(i,j) = 0; break; }
		    if(c=='1') {image(i,j) = 1; break; }
		    throw "P1: bad format";
		}
	    }
	} else if(ptype=='2') {
	    image.resize(w,h);
	    for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
		int v;
		if(fscanf(stream,"%d",&v)!=1) throw "P2: bad format";
		image(i,j) = v;
	    }
	} else if(ptype=='3') {
	    for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
		int value = 0;
		for(int k=0;k<3;k++) {
		    int v;
		    if(fscanf(stream,"%d",&v)!=1) throw "P3: bad format";
		    value += v;
		}
		image(i,j) = value/3;
	    }
	} else if(ptype=='4') {
	    image.resize(w,h);
	    int bit = 8;
	    int c = 0;
	    for(int j=h-1;j>=0;j--) {
		for(int i=0;i<w;i++) {
		    if(bit>7) {
			bit=0;
			c = safe_getc(stream); 
		    }
		    image(i,j) = (c&0x80)?0:255;
		    c<<=1; bit++;
		}
		if(bit%8!=0) {
		    c = safe_getc(stream);
		    bit = 0;
		}
	    }
	} else if(ptype=='5') {
	    image.resize(w,h);
	    for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
		int c = safe_getc(stream);
		image(i,j) = c;
	    }
	} else if(ptype=='6') {
	    image.resize(w,h);
	    for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
		int c = safe_getc(stream) + safe_getc(stream) + safe_getc(stream);
		image(i,j) = c/3;
	    }
	} else {
	    throw "PNM: unknown type";
	}
    }

    void read_pnm(FILE *stream,array2<int> &image) {
	array2<unsigned char> temp;
	read_pnm(stream,temp);
	int w = temp.dim(0), h = temp.dim(1);
	image.resize(w,h);
	for(int i=0;i<w;i++) for(int j=0;j<h;j++) image(i,j) = temp(i,j);
    }

    void write_pgm(FILE *stream,array2<unsigned char> &image) {
	int w = image.dim(0), h = image.dim(1);
	fprintf(stream,"P5\n%d %d\n%d\n",w,h,255);
	for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
	    safe_putc(image(i,j),stream);
	}
    }

    void write_ppm(FILE *stream,array2<int> &image) {
	int w = image.dim(0), h = image.dim(1);
	fprintf(stream,"P6\n%d %d\n%d\n",w,h,255);
	for(int j=h-1;j>=0;j--) for(int i=0;i<w;i++) {
	    safe_putc(image(i,j)>>16,stream);
	    safe_putc(image(i,j)>>8,stream);
	    safe_putc(image(i,j),stream);
	}
    }

    void write_pbm(FILE *stream,array2<unsigned char> &image) {
	int w = image.dim(0), h = image.dim(1);
	fprintf(stream,"P4\n%d %d\n",w,h);
	int bit = 7;
	int c = 0;
	for(int j=h-1;j>=0;j--) {
	    for(int i=0;i<w;i++) {
		if(image(i,j)) c |= (1<<bit);
		bit--;
		if(bit<0) {
		    safe_putc(c,stream);
		    c = 0;
		    bit = 7;
		}
	    }
	    if(bit!=7) {
		safe_putc(c,stream);
		c = 0;
		bit = 7;
	    }
	}
	if(bit<7) safe_putc(c,stream);
    }
}
