// -*- C++ -*- Copyright (c) 1990-1995 by Thomas M. Breuel

////////////////////////////////////////////////////////////////
// I/O related code
////////////////////////////////////////////////////////////////
 
#ifndef h_coio__
#define h_coio__

#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <stdio.h>

namespace COLIB {

    // I/O and environment related utility functions

    inline double now() {
	struct timeval tv;
	gettimeofday(&tv,0);
	return tv.tv_sec + 1e-6 * tv.tv_usec;
    }

    struct stdio {
	FILE *stream;
	stdio() {
	    stream = 0;
	}
	stdio(char *name,char *mode) {
	    stream = 0;
	    open(name,mode);
	}
	void open(char *name,char *mode) {
	    close();
	    stream = fopen(name,mode);
	    if(!stream) throw "fopen failed";
	}
	void close() {
	    if(!stream) return;
	    fclose(stream);
	    stream = 0;
	}
	void popen(char *name,char *mode) {
	    close();
	    stream = ::popen(name,mode);
	    if(!stream) throw "fopen failed";
	}
	void pclose() {
	    if(!stream) return;
	    ::pclose(stream);
	    stream = 0;
	}
	operator FILE*() {
	    return stream;
	}
	operator bool() {
	    return !!stream;
	}
	void flush() {
	    fflush(stream);
	}
    };

    inline int igetenv(char *name,int dflt) {
	int result = getenv(name)?atoi(getenv(name)):dflt;
	int where = 0;
	if(strcmp(name,"verbose_params")) where = igetenv("verbose_params",0);
	switch(where) {
	case 1: fprintf(stdout,"__param__ %s = %d\n",name,result); break;
	case 2: fprintf(stderr,"__param__ %s = %d\n",name,result); break;
	default: ;
	}
	return result;
    }

    inline float fgetenv(char *name,float dflt) {
	float result = getenv(name)?atof(getenv(name)):dflt;
	int where = igetenv("verbose_params",0);
	switch(where) {
	case 1: fprintf(stdout,"__param__ %s = %g\n",name,result); break;
	case 2: fprintf(stderr,"__param__ %s = %g\n",name,result); break;
	default: ;
	}
	return result;
    }

    inline double dgetenv(char *name,double dflt) {
	double result = getenv(name)?atof(getenv(name)):dflt;
	int where = igetenv("verbose_params",0);
	switch(where) {
	case 1: fprintf(stdout,"__param__ %s = %g\n",name,result); break;
	case 2: fprintf(stderr,"__param__ %s = %g\n",name,result); break;
	default: ;
	}
	return result;
    }

    inline char *sgetenv(char *name,char *dflt) {
	char *result = getenv(name)?getenv(name):dflt;
	int where = igetenv("verbose_params",0);
	switch(where) {
	case 1: fprintf(stdout,"__param__ %s = %s\n",name,result); break;
	case 2: fprintf(stderr,"__param__ %s = %s\n",name,result); break;
	default: ;
	}
	return result;
    }

    inline int mkseed() {
	int seed = igetenv("seed",-1);
	if(seed!=-1) return seed;
	return abs(getpid()+int(now()*1000.0));
    }

    void print_timing(const char *name,double s) {
	int min = (int)floor(s/60.0); 
	fprintf(stderr,"%s\t%dm%5.3fs\n",name,min,s-60.0*min);
    }

} // namespace colib

#endif
