// -*- C++ -*- Copyright (c) 1990-1995 by Thomas M. Breuel

////////////////////////////////////////////////////////////////
// Array classes and related code
////////////////////////////////////////////////////////////////

#ifndef h_colib__
#define h_colib__

#undef COASSERT
#undef CORANGE

#ifndef UNSAFE
#define ASSERT(X) do{if(!(X)) throw __FILE__ ": assertion failed " #X;}while(0)
#define COASSERT(X) do{if(!(X)) throw __FILE__ ": assertion failed " #X;}while(0)
#define CORANGE(I,N) do{if(unsigned(I)>=unsigned(N)) throw __FILE__ ": range check failed"; }while(0)
#define RANGE(I,N) do{if(unsigned(I)>=unsigned(N) || I<0) throw __FILE__ ": range check failed"; }while(0) 
#else
#define COASSERT(X) do{}while(0)
#define CORANGE(I,N) do{}while(0)
#define RANGE(I,N) do{}while(0)
#define ASSERT(X) do{}while(0)
#endif

#include <stdio.h>

namespace COLIB {
    // 1D arrays

    template <class T>
    class array {
    protected:
	int size;
	T *data;
    public:
	array() {
	    data = 0;
	    size = 0;
	}
	array(int n) {
	    data = new T[n];
	    size = n;
	}
	~array() {
	    delete [] data;
	    data = 0;
	    size = -1;
	}
	void resize(int n) {
	    delete [] data;
	    data = new T[n];
	    size = n;
	}
	int length() {
	    return size;
	}
	void operator=(const T &value) {
	    for(int i=0;i<size;i++) data[i] = value;
	}
	template <class S>
	void operator=(const array<S> &source) {
	    resize(source.size);
	    S *sdata = &source.data[0];
	    for(int i=0;i<size;i++) data[i] = (T)sdata[i];
	}
	T &operator[](int i) const {
	    CORANGE(i,size);
	    return data[i];
	}
	T &operator()(int i) const {
	    CORANGE(i,size);
	    return data[i];
	}
	int rank() {
	    return 1;
	}
	int dim(int i) const {
	    CORANGE(i,1);
	    return size;
	}
    };

    // 2D arrays

    template <class T>
    class array2 {
    protected:
	int d0,d1;
	T *data;
    public:
	array2() {
	    d0 = d1 = 0;
	    data = 0;
	}
	array2(int d0,int d1):d0(d0),d1(d1) {
	    data = new T[d0*d1];
	}
	~array2() {
		delete [] data;
	    data = 0;
		d0 = 0;
	    d1 = 0;
	}
	int dim(int i) const {
	    switch(i){
		case 0: return d0;
		case 1: return d1;
		default: throw "bad dimension";
	    }
	}
	void resize(int d0,int d1) {
	    this->d0 = d0;
	    this->d1 = d1;
	    if(data) delete [] data;
	    data = new T[d0*d1];
	}
	void operator=(const T &value) {
	    int size = d0*d1;
	    for(int i=0,total = size;i<total;i++) data[i] = value;
	}
	template <class S>
	void operator=(const array2<S> &source) {
	    resize(source.dim(0),source.dim(1));
	    int size = d0*d1;
	    S *sdata = &source(0,0);
	    for(int i=0;i<size;i++) data[i] = (T)sdata[i];
	}
	T &operator()(int i,int j) const {
	    CORANGE(i,d0);
	    CORANGE(j,d1);
//	    fprintf (stderr, "HUA!");
	    return data[i*d1+(d1-1-j)];
//	    return data[i*d1+j];
	}
	T max() {
	    COASSERT(d0*d1>0);
	    T mv = data[0];
	    for(int i=1,total=d0*d1;i<total;i++) {
		if(data[i]>mv) mv = data[i];
	    }
	    return mv;
	}
	T min() {
	    COASSERT(d0*d1>0);
	    T mv = data[0];
	    for(int i=1,total=d0*d1;i<total;i++) {
		if(data[i]<mv) mv = data[i];
	    }
	    return mv;
	}
	template <class S>
	void get1d(int axis,array<S> &out,int i) {
	    if(axis==0) {
		out.resize(d1);
		for(int j=0;j<d1;j++) {
		    out(j) = (S)operator()(i,j);
		}
	    } else if(axis==1) {
		out.resize(d0);
		for(int j=0;j<d0;j++) {
		    out(j) = (S)operator()(j,i);
		}
	    }
	}
	template <class S>
	void put1d(int axis,array<S> &in,int i) {
	    if(axis==0) {
		for(int j=0;j<d1;j++) {
		    operator()(i,j) = (T)in(j);
		}
	    } else if(axis==1) {
		for(int j=0;j<d0;j++) {
		    operator()(j,i) = (T)in(j);
		}
	    }
	}
	int rank() {
	    return 2;
	}
    };

    // 3D arrays

    template <class T>
    class array3 {
    protected:
	int d0,d1,d2;
	T *data;
    public:
	array3() {
	    d0 = d1 = d2 = 0;
	    data = 0;
	}
	array3(int d0,int d1,int d2):d0(d0),d1(d1) {
	    data = new T[d0*d1*d2];
	}
	~array3() {
	    delete [] data;
	    d0 = 0;
	    d1 = 0;
	    d2 = 0;
	}
	int dim(int i) const {
	    switch(i){
	    case 0: return d0;
	    case 1: return d1;
	    case 2: return d2;
	    default: OOPS;
	    }
	}
	void resize(int d0,int d1,int d2) {
	    this->d0 = d0;
	    this->d1 = d1;
	    this->d2 = d2;
	    if(data) delete [] data;
	    data = new T[d0*d1*d2];
	}
	void operator=(const T &value) {
	    int size = d0*d1*d2;
	    for(int i=0,total = size;i<total;i++) data[i] = value;
	}
	template <class S>
	void operator=(const array3<S> &source) {
	    resize(source.dim(0),source.dim(1),source.dim(2));
	    int size = d0*d1*d2;
	    T *sdata = &source.data(0,0,0);
	    for(int i=0;i<size;i++) data[i] = (T)sdata[i];
	}
	T &operator()(int i,int j,int k) const {
	    CORANGE(i,d0);
	    CORANGE(j,d1);
	    CORANGE(k,d2);
	    return data[i*d1*d2+j*d2+k];
	}
	int rank() {
	    return 3;
	}
    };

    // stack data class--1D array with push/pop operations

    template <class T,int DFLT=0>
    class stack {
    protected:
	T *data;
	int fill;
	int total;
    public:
	stack() {
	    if(DFLT==0) {
		data = 0;
		fill = 0;
		total = 0;
	    } else {
		data = new T[DFLT];
		fill = 0;
		total = DFLT;
	    }
	}
	explicit stack(int n) {
	    data = new T[n];
	    fill = 0;
	    total = n;
	}
	~stack() {
	    reset();
	}
	void clear() {
	    fill = 0;
	}
	void reset() {
	    if(data) delete [] data;
	    data = 0;
	    fill = 0;
	    total = 0;
	}
	int length() {
	    return fill;
	}
	T &push() {
	    if(fill==total) growto(2*total);
	    return data[fill++];
	}
	void push(T &value) {
	    push() = value;
	}
	void push(const T &value) {
	    push() = value;
	}
	T &pop() {
	    ASSERT(fill>=1);
	    return data[--fill];
	}
	T &at(int i) {
	    RANGE(i,fill);
	    return data[i];
	}
	T &unsafe_at(int i) {
	    RANGE(i,fill);
	    return data[i];
	}
	T &operator[](int i) {
	    RANGE(i,fill);
	    return data[i];
	}
	void resize(int n) {
	    if(n>total) growto(n);
	    fill = n;
	}
	void operator=(T value) {
	    for(int i=0;i<fill;i++) data[i] = value;
	}
	template <class S>
	void copy(stack<S> &other) {
	    clear();
	    for(int i=0;i<other.fill;i++) push(other[i]);
	}
	template <class S>
	void copy(const stack<S> &other) {
	    clear();
	    for(int i=0;i<other.fill;i++) push(other[i]);
	}
	void growto(int ntotal) {
	    ntotal = ntotal>4? ntotal : 4;
	    T *ndata = new T[ntotal];
	    if(data) {
		for(int i=0;i<fill;i++) ndata[i] = data[i];
		delete [] data;
	    }
	    data = ndata;
	    total = ntotal;
	}
    private:
	stack(stack<T> &other);
	stack(const stack<T> &other);
    };

    // smart pointer that automatically deletes the object in
    // question when it goes out of scope

    template <class T>
    class autodel {
    protected:
	T *data;
    public:
	autodel() {
	    data = 0;
	}
	~autodel() {
	    if(data) delete data;
	}
	autodel(T *other) {
	    data = other;
	}
	void operator=(T *other) {
	    if(data) delete data;
	    data = other;
	}
	T &operator*() {
	    return *data;
	}
	T *operator->() {
	    return data;
	}
	operator bool() {
	    return !!p;
	}
	T *steal() {
	    T *result = data;
	    data = 0;
	    return result;
	}
    };

    // smart pointer that automatically allocates objects
    // when referenced and deletes them when it goes out of scope

    template <class T>
    class autoobj {
    protected:
	T *data;
    public:
	autoobj() {
	    data = 0;
	}
	~autoobj() {
	    if(data) delete data;
	}
	autoobj(T *other) {
	    data = other;
	}
	void operator=(T *other) {
	    if(data) delete data;
	    data = other;
	}
	T &operator*() {
	    if(!data) data = new T();
	    return *data;
	}
	T *operator->() {
	    if(!data) data = new T();
	    return data;
	}
	operator bool() {
	    return !!p;
	}
	T *steal() {
	    T *result = data;
	    data = 0;
	    return result;
	}
    };

    // reference counting class
    // also automatically allocates object as needed

    template <class T,bool AUTOALLOC=false>
    class counted {
    protected:
	struct TC : T {
	    int refcount_;
	};
	TC *p;
    public:
	counted() {
	    p = 0;
	}
	counted(const counted<T> &other) {
	    other.incref();
	    p = other.p;
	}
	counted(counted<T> &other) {
	    other.incref();
	    p = other.p;
	}
	~counted() {
	    decref();
	    p = 0;
	}
	void operator=(counted<T> &other) {
	    other.incref();
	    decref();
	    p = other.p;
	}
	void operator=(const counted<T> &other) {
	    other.incref();
	    decref();
	    p = other.p;
	}
	void steal(counted<T> &other) {
	    other.incref();
	    decref();
	    p = other.p;
	    other.drop();
	}
	void steal(const counted<T> &other) {
	    other.incref();
	    decref();
	    p = other.p;
	    other.drop();
	}
	bool allocated() {
	    return !p;
	}
	void allocate() {
	    p = new TC();
	    p->refcount_ = 1;
	}
	operator bool() {
	    return !!p;
	}
	void drop() {
	    decref();
	    p = 0;
	}
	T &operator *() {
	    if(!p) {if(AUTOALLOC) allocate(); else throw "object not allocated";}
	    return *(T*)p;
	}
	T *operator->() {
	    if(!p) {if(AUTOALLOC) allocate(); else throw "object not allocated";}
	    return (T*)p;
	}
	operator T&() {
	    if(!p) {if(AUTOALLOC) allocate(); else throw "object not allocated";}
	    return *(T*)p;
	}
	operator T*() {
	    if(!p) {if(AUTOALLOC) allocate(); else throw "object not allocated";}
	    return (T*)p;
	}
	void incref() const {
	    check();
	    if(p) {
		if(p->refcount_>10000000) abort();
		if(p->refcount_<0) abort();
		p->refcount_++;
	    }
	}
	void decref() const {
	    check();
	    if(p) {
		if(--p->refcount_==0) delete p;
		((counted<T>*)this)->p = 0;
	    }
	}
	void check() const {
#ifndef UNSAFE
	    if(!p) return;
	    if(p->refcount_>10000000) abort();
	    if(p->refcount_<0) abort();
#endif
	}
    };

    template <class T>
    void swap(counted<T> &a,counted<T> &b) {
	typename counted<T>::TC *temp = a.p;
	a.p = b.p;
	b.p = temp;
    }



} // namespace colib

#undef CORANGE
#undef COASSERT

#endif
