/**
  *  An evolutionary approach to reconstruct 3D binary images by their projections
  *  SSIP 2009 -- Team 2 -- Project 11: Binary Tomography
 **/

#ifndef __evobintom3D_h__
#define __evobintom3D_h__

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

/* define logical true and false */
#define TRUE (1==1)
#define FALSE !TRUE
/* the minimum number of the sphares */
#define MIN_GEN_SPHARES 1
/* the maximum number of the sphares */
#define MAX_GEN_SPHARES 10


/* this macro generates a random number between the two given integers */
#define getrandom( min, max ) ((rand() % (int)(((max)+1) - (min))) + (min))


/* 
 * to store projection data. Only store projections from three angles in 3D.
 */
typedef int Proj;

/*
 * Structure to store the parameters of a Sphare.
 * x - x-coordinate
 * y - y-coordinate
 * z - z-coordinate
 * r - the radius
 */
typedef struct Sphare{
    int x;
    int y;
    int z;
    int r;
} Sphare;

/*
 * Strucure to store the configuration of the sphares. 
 * nrObjects - the number of objects, which are on the image.
 * data      - the array to store the Sphares
 */
typedef struct Conf{
   int nrObjects;
   Sphare *data;
} Conf;

/* global variables */
static int picSize;
static int worstfitness;
Proj   *tmpProj;
int    ***picture;
Conf   *nG = NULL; /* new Generation*/
int    *newfitness = NULL;
int    worstplace;
double pos_CO;
double pos_m_nr;
double pos_m_move;
double pos_m_size;


/*
 * initialize the program and memory allcation for the global variables
 */
int init(){
    int i, j;
    srand(time(NULL));

    tmpProj = (int*)malloc(3*picSize*picSize*sizeof(int));
    if (!tmpProj){ printf("Can't allocate memory for temporary projections.\n"); return FALSE; }

    picture = (int***)malloc(picSize*sizeof(int**));
    if (!picture){ printf("Can't allocate memory for picture.\n"); return FALSE; }
    for (i=0; i<picSize; i++){
        picture[i] = (int**)malloc(picSize*sizeof(int*));
        if (!picture[i]){ printf("Can't allocate memory for picture.\n"); return FALSE; }
        for (j=0; j<picSize; j++){
            picture[i][j] = (int*)malloc(picSize*sizeof(int));
            if (!picture[i][j]){ printf("Can't allocate memory for picture.\n"); return FALSE; }
        }
    }

    return TRUE;
}

/*
 * read input projections from input file 
 * fileName - input file name
 * proj     - the read projection data
 * returns: the pointer to the proj on success
 *          null pointer on failure
 */
Proj* readInput(char* fileName) {
    FILE *f = NULL;
    int i;
    Proj* proj;

    f = fopen(fileName, "r");
    if (!f) { printf("Can't open file.\n"); return NULL; }

    if (1 != fscanf(f, "%d", &picSize)){ printf("Can't read picture size.\n"); return NULL; }
    
    if (picSize <= 0) { printf("Wrong picture size.\n"); return NULL; }

    proj = (Proj*)malloc(3*picSize*picSize*sizeof(int));
    if (!proj){ printf("Can't allocate memory for projection data.\n"); return NULL; }
    
    for (i=0; i<3*picSize*picSize; i++){
        if (1 != fscanf(f, "%d", &proj[i])){ printf("Can't read projection data.\n"); return NULL; }
    }

    return proj;
}

/* 
 * generate initial population 
 * population - array to store the configuration pointers
 * popSize    - the size of the population
 * picSize    - the size of the three dimensional cubic picture
 * returns: TRUE on success
 *          FALSE on failure
 */
int genInitPop(Conf* population, int popSize){
    int i, j;

    for (i=0; i<popSize; i++) {
        population[i].nrObjects = getrandom(MIN_GEN_SPHARES, MAX_GEN_SPHARES);
        population[i].data = (Sphare*)malloc(population[i].nrObjects*sizeof(Sphare));
        for (j=0; j<population[i].nrObjects; j++){
            population[i].data[j].x = getrandom(0, picSize-1);
            population[i].data[j].y = getrandom(0, picSize-1);
            population[i].data[j].z = getrandom(0, picSize-1);
            population[i].data[j].r = getrandom(5, picSize/2);
        }
    }

    return TRUE;
}

/*
 * Make the projections of a configuration
 * conf    - the configuration
 * proj    - output: the projections
 * picSize - the three dimensional cubic picture size
 */
void makeProjs(Conf *conf, Proj* proj){
    int i, j, k, l;
    int start_i, end_i, start_j, end_j, start_k, end_k;
    
    /* reset the picture */
    for (i=0; i<picSize; i++){
        for (j=0; j<picSize; j++){
            for (k=0; k<picSize; k++){
                picture[i][j][k] = 0;
                /* reset the projections */
                if (k<3) proj[i+j*picSize+k*picSize*picSize] = 0;
            }
        }
    }

    /* for all object we set those positions of the picture to 1 which are 
       closer to the object center than r */
    for (l=0; l<conf->nrObjects; l++){
        start_i = (conf->data[l].x - conf->data[l].r < 0 ? 0 : (conf->data[l].x - conf->data[l].r));
        end_i   = (conf->data[l].x + conf->data[l].r < picSize ? (conf->data[l].x + conf->data[l].r) : picSize);
        start_j = (conf->data[l].y - conf->data[l].r < 0 ? 0 : (conf->data[l].y - conf->data[l].r));
        end_j   = (conf->data[l].y + conf->data[l].r < picSize ? (conf->data[l].y + conf->data[l].r) : picSize);
        start_k = (conf->data[l].z - conf->data[l].r < 0 ? 0 : (conf->data[l].z - conf->data[l].r));
        end_k   = (conf->data[l].z + conf->data[l].r < picSize ? (conf->data[l].z + conf->data[l].r) : picSize);
/* printf("%d %d - %d %d - %d %d\n", start_i, end_i, start_j, end_j, start_k, end_k); */
        for (i=start_i; i<end_i; i++){
            for (j=start_j; j<end_j; j++){
                for (k=start_k; k<end_k; k++){
                    if (((i - conf->data[l].x)*(i - conf->data[l].x) +
                         (j - conf->data[l].y)*(j - conf->data[l].y) +
                         (k - conf->data[l].z)*(k - conf->data[l].z)) <=
                         ((conf->data[l].r)*(conf->data[l].r))){
                        picture[i][j][k] = 1;
                    }                                 
                }
            }
        }
    }
    
    /* calculating the projections */
    for (i=0; i<picSize; i++){
        for (j=0; j<picSize; j++){
            for (k=0; k<picSize; k++){
                proj[0*picSize*picSize + i*picSize + j] += picture[i][j][k];
                proj[1*picSize*picSize + j*picSize + k] += picture[j][k][i];
                proj[2*picSize*picSize + k*picSize + i] += picture[k][i][j];
            }
        }
    }
}

/*
 * calculate fitness value
 * conf - configuration of Sphares
 * proj - input projections
 * return: the fitness value
 */
int calcFitness(Conf* conf, Proj* proj){
    int fitness = 0;
    int i;

    makeProjs(conf, tmpProj);
    for (i=0; i<3*picSize*picSize; i++){
        fitness += abs(proj[i] - tmpProj[i]);
    }

    return fitness;
}

/* 
 * write the configuration to file
 * filename - the file to write in
 * conf     - the configuration
 * returns: TRUE on success
 *          FALSE on failure
 */
int writeConf(char* filename, Conf* conf){
    FILE *f;
    int i;

    if (!(f = fopen(filename, "w"))) { printf("Can't open output file.\n"); return FALSE;}

    fprintf(f, "%d\n", conf->nrObjects);
    for (i=0; i<conf->nrObjects; i++) {
        fprintf(f, "%d %d %d %d\n", conf->data[i].x, conf->data[i].y, conf->data[i].z, conf->data[i].r);
    }
    return TRUE;
}

/* 
 * write the projections to file 
 * filename - the file to write in
 * p        - the projections
 * returns: TRUE on success
 *          FALSE on failure
 */
int writeProjs(char *filename, Proj *p) {
    FILE *f;
    int i;

    if (!(f = fopen(filename, "w"))) { printf("Can't open output file.\n"); return FALSE;}
    /* first write the picture size */
    fprintf(f, "%d\n", picSize);
    /* print the data */
    for (i=0; i<3*picSize*picSize; i++){
        fprintf(f, "%d ", p[i]);
    }

    fclose(f);

    return TRUE;
}

/*
 * nextGeneration - the main evolutionary step (do a complete generation/iteration)
 * inputProj   - the projections of the original image
 * pop         - the previous population
 * fitnes      - the fitness of the prevous population
 * popSize     - the size of the population
 * bestfitness - the best fitness value
 * bestitem    - the number of the best fitness value
 * returns: TRUE on success
 *          FALSE on failure
 */
int nextGeneration(Proj *inputProj, Conf* pop, int* fitness, int popSize, int *bestfitness, int *bestitem){
    int nrOffsp = 0; /* number of generated offsprings */
    int random, r1, r2;
    int i, j;

    if (!nG){nG = (Conf*)malloc(6*popSize*sizeof(Conf));
        if (!nG){printf("Can't allocate memory for nG.\n"); return FALSE;}}
    if (!newfitness){newfitness = (int*)malloc(6*popSize*sizeof(int));
        if (!newfitness){printf("Can't allocate memory for newfitness.\n"); return FALSE;}}

    for (i=0; i<popSize; i++){
        /* crossover */
        if ((rand()/(double)RAND_MAX)<pos_CO){
            do {
                random = getrandom(0, popSize-1);
            }while (random == i);
            r1 = getrandom(0, pop[i].nrObjects-1);        /* we cut the first list here */
            r2 = getrandom(0, pop[random].nrObjects-1);   /* we cut the second list here*/
            if ((r1+r2)>0){
                nrOffsp++;
                nG[nrOffsp-1].data = (Sphare*)malloc((r1+r2)*sizeof(Sphare));
                nG[nrOffsp].data = (Sphare*)malloc((pop[i].nrObjects-r1 + pop[random].nrObjects-r2)*sizeof(Sphare));
                nG[nrOffsp-1].nrObjects = r1+r2;
                nG[nrOffsp].nrObjects = (pop[i].nrObjects-r1 + pop[random].nrObjects-r2);

                for (j=0; j<r1; j++){
                    nG[nrOffsp-1].data[j].x = pop[i].data[j].x;
                    nG[nrOffsp-1].data[j].y = pop[i].data[j].y;
                    nG[nrOffsp-1].data[j].z = pop[i].data[j].z;
                    nG[nrOffsp-1].data[j].r = pop[i].data[j].r;
                }
                for (j=0; j<r2; j++) {
                    nG[nrOffsp-1].data[j+r1].x = pop[random].data[j].x;
                    nG[nrOffsp-1].data[j+r1].y = pop[random].data[j].y;
                    nG[nrOffsp-1].data[j+r1].z = pop[random].data[j].z;
                    nG[nrOffsp-1].data[j+r1].r = pop[random].data[j].r;
                }
                for (j=r1; j<pop[i].nrObjects; j++){
                    nG[nrOffsp].data[j-r1].x = pop[i].data[j].x;
                    nG[nrOffsp].data[j-r1].y = pop[i].data[j].y;
                    nG[nrOffsp].data[j-r1].z = pop[i].data[j].z;
                    nG[nrOffsp].data[j-r1].r = pop[i].data[j].r;
                }
                for (j=r2; j<pop[random].nrObjects; j++) {
                    nG[nrOffsp].data[j-r2+pop[i].nrObjects-r1].x = pop[random].data[j].x;
                    nG[nrOffsp].data[j-r2+pop[i].nrObjects-r1].y = pop[random].data[j].y;
                    nG[nrOffsp].data[j-r2+pop[i].nrObjects-r1].z = pop[random].data[j].z;
                    nG[nrOffsp].data[j-r2+pop[i].nrObjects-r1].r = pop[random].data[j].r;
                }
                newfitness[nrOffsp-1] = calcFitness(&nG[nrOffsp-1], inputProj);
                newfitness[nrOffsp] = calcFitness(&nG[nrOffsp], inputProj);
                /* the goodness is only checked for the second offspring (technically more simply) */
                if (newfitness[nrOffsp]<worstfitness)
                    nrOffsp++;
                else
                    free(nG[nrOffsp].data);
            }
        }

        /* mutation */
        if ((rand()/(double)RAND_MAX)<pos_m_nr){
            if ((rand()/(double)RAND_MAX)<0.5){
                /* increase */
                nG[nrOffsp].data = (Sphare*)malloc((pop[i].nrObjects+1)*sizeof(Sphare));
                nG[nrOffsp].nrObjects = pop[i].nrObjects+1;
                for (j=0; j<pop[i].nrObjects; j++) {
                    nG[nrOffsp].data[j].x = pop[i].data[j].x;
                    nG[nrOffsp].data[j].y = pop[i].data[j].y;
                    nG[nrOffsp].data[j].z = pop[i].data[j].z;
                    nG[nrOffsp].data[j].r = pop[i].data[j].r;
                }
                j=pop[i].nrObjects;
                nG[nrOffsp].data[j].x = getrandom(0, picSize-1);
                nG[nrOffsp].data[j].y = getrandom(0, picSize-1);
                nG[nrOffsp].data[j].z = getrandom(0, picSize-1);
                nG[nrOffsp].data[j].r = getrandom(5, picSize/2);

                newfitness[nrOffsp] = calcFitness(&nG[nrOffsp], inputProj);
                if (newfitness[nrOffsp]<worstfitness)
                    nrOffsp++;
                else
                    free(nG[nrOffsp].data);
            }
            else
            {
                /* decrease */
                if (pop[i].nrObjects>1){
                    nG[nrOffsp].data = (Sphare*)malloc((pop[i].nrObjects-1)*sizeof(Sphare));
                    nG[nrOffsp].nrObjects = pop[i].nrObjects-1;
                    random = getrandom(0, pop[i].nrObjects-1);
                    for (j=0; j<random; j++) {
                        nG[nrOffsp].data[j].x = pop[i].data[j].x;
                        nG[nrOffsp].data[j].y = pop[i].data[j].y;
                        nG[nrOffsp].data[j].z = pop[i].data[j].z;
                        nG[nrOffsp].data[j].r = pop[i].data[j].r;
                    }
                    for (j=random+1; j<pop[i].nrObjects; j++) {
                        nG[nrOffsp].data[j-1].x = pop[i].data[j].x;
                        nG[nrOffsp].data[j-1].y = pop[i].data[j].y;
                        nG[nrOffsp].data[j-1].z = pop[i].data[j].z;
                        nG[nrOffsp].data[j-1].r = pop[i].data[j].r;
                    }
                    newfitness[nrOffsp] = calcFitness(&nG[nrOffsp], inputProj);
                    if (newfitness[nrOffsp]<worstfitness)
                        nrOffsp++;
                    else
                        free(nG[nrOffsp].data);
                }
            }
        }

        /* position */
        if ((rand()/(double)RAND_MAX)<pos_m_move){
            nG[nrOffsp].data = (Sphare*)malloc(pop[i].nrObjects*sizeof(Sphare));
            nG[nrOffsp].nrObjects = pop[i].nrObjects;
            if (pop[i].nrObjects>1) random = getrandom(0, pop[i].nrObjects-1);
            else random = 0;
            for (j=0; j<pop[i].nrObjects; j++) {
                if (j==random){
                    nG[nrOffsp].data[j].x = getrandom(pop[i].data[j].x-15, pop[i].data[j].x+15);
                    if (nG[nrOffsp].data[j].x<0)        nG[nrOffsp].data[j].x = 0;
                    if (nG[nrOffsp].data[j].x>=picSize) nG[nrOffsp].data[j].x = picSize-1;
            
                    nG[nrOffsp].data[j].y = getrandom(pop[i].data[j].y-15, pop[i].data[j].y+15);
                    if (nG[nrOffsp].data[j].y<0)        nG[nrOffsp].data[j].y = 0;
                    if (nG[nrOffsp].data[j].y>=picSize) nG[nrOffsp].data[j].y = picSize-1;
            
                    nG[nrOffsp].data[j].z = getrandom(pop[i].data[j].z-15, pop[i].data[j].z+15);
                    if (nG[nrOffsp].data[j].z<0)        nG[nrOffsp].data[j].z = 0;
                    if (nG[nrOffsp].data[j].z>=picSize) nG[nrOffsp].data[j].z = picSize-1;
                }
                else {
                    nG[nrOffsp].data[j].x = pop[i].data[j].x;
                    nG[nrOffsp].data[j].y = pop[i].data[j].y;
                    nG[nrOffsp].data[j].z = pop[i].data[j].z;
                }
                nG[nrOffsp].data[j].r = pop[i].data[j].r;
            }
            newfitness[nrOffsp] = calcFitness(&nG[nrOffsp], inputProj);
            if (newfitness[nrOffsp]<worstfitness) 
                nrOffsp++;
            else
                free(nG[nrOffsp].data);
        }

        /* size */
        if ((rand()/(double)RAND_MAX)<pos_m_size){
            nG[nrOffsp].data = (Sphare*)malloc(pop[i].nrObjects*sizeof(Sphare));
            nG[nrOffsp].nrObjects = pop[i].nrObjects;
            if (pop[i].nrObjects>1) random = getrandom(0, pop[i].nrObjects-1);
            else random = 0;
            for (j=0; j<pop[i].nrObjects; j++) {
                if (j==random){
                    nG[nrOffsp].data[j].r = getrandom(pop[i].data[j].r-15, pop[i].data[j].r+15);
                    if (nG[nrOffsp].data[j].r<5)        nG[nrOffsp].data[j].r = 5;
                    if (nG[nrOffsp].data[j].r>=picSize) nG[nrOffsp].data[j].r = picSize-1;
                }
                else {
                    nG[nrOffsp].data[j].r = pop[i].data[j].r;
                }
                nG[nrOffsp].data[j].x = pop[i].data[j].x;
                nG[nrOffsp].data[j].y = pop[i].data[j].y;
                nG[nrOffsp].data[j].z = pop[i].data[j].z;
            }
            newfitness[nrOffsp] = calcFitness(&nG[nrOffsp], inputProj);
            if (newfitness[nrOffsp]<worstfitness)
                nrOffsp++;
            else
                free(nG[nrOffsp].data);
        }
    } /* for */
    /* selection */
    for (i=0; i<nrOffsp; i++) {
        /* put the i th element from the new population to the j th place in the base population */
        if (newfitness[i]<worstfitness)
        {
            if (pop[worstplace].data){
                free(pop[worstplace].data);
            }
            pop[worstplace].nrObjects = nG[i].nrObjects;
            pop[worstplace].data = nG[i].data;
            fitness[worstplace] = newfitness[i];
            /*worstplace means the prevous worst place */
            if (newfitness[i]<*bestfitness){ *bestfitness = fitness[worstplace]; *bestitem = worstplace;}

            /* refresh the worth */
            worstfitness = 0;
            for (j=0; j<popSize; j++) {
                if (fitness[j]>worstfitness){
                    worstfitness = fitness[j];
                    worstplace = j;
                }
            }
        }
        else{
            free (nG[i].data);
        }
    }
    return TRUE;
}

/*
 * simplify the configuration - remove objects, which are totally inside an other one 
 * pop     - population
 * popSize - size of the population
 * returns: TRUE on success
 *          FALSE on failure
 */
int simplify(Conf *pop, int popSize){
    int i, j, k;
    double dist;
    Sphare *tmpData;
    int *list;
    int nr_in_list;
    int next_in_list;

    /* for every images we see where is overlapping between the sphares and after we 
       delete the smaller one */
    for (i=0; i<popSize; i++){
        nr_in_list=0;
        list = (int*)malloc(pop[i].nrObjects*sizeof(int));
        if (!list){printf("Can't allocate memory for list of deletable items.\n"); return FALSE;}
        for (j=0; j<pop[i].nrObjects; j++){
            for (k=0; k<pop[i].nrObjects; k++){
                if (pop[i].data[j].r<=pop[i].data[k].r && j!=k){
                    /* only deal with the squares of the distances */
                    dist = (pop[i].data[j].x-pop[i].data[k].x)*(pop[i].data[j].x-pop[i].data[k].x)+
                           (pop[i].data[j].y-pop[i].data[k].y)*(pop[i].data[j].y-pop[i].data[k].y)+
                           (pop[i].data[j].z-pop[i].data[k].z)*(pop[i].data[j].z-pop[i].data[k].z);
                    dist = sqrt(dist);
                    /* if  d+r1<=r2 than sign the smaller circle (r1) for deleting */
                    if (dist+pop[i].data[j].r <= pop[i].data[k].r){
                        list[nr_in_list] = j;
                        nr_in_list++;
                        break;
                    }
                }
            }
        }
        /* delete the not necessarry sphares */
        if (nr_in_list>0){
            tmpData = (Sphare*)malloc((pop[i].nrObjects-nr_in_list)*sizeof(Sphare));
            if (!tmpData){printf("Can't allocate memory for temporary data.\n"); return FALSE;}
            k=0;
            next_in_list = 0;
            for (j=0; j<pop[i].nrObjects; j++){
                if (next_in_list<nr_in_list){
                    if (j == list[next_in_list]){
                        next_in_list++;
                    }
                    else {
                        tmpData[k].x = pop[i].data[j].x;
                        tmpData[k].y = pop[i].data[j].y;
                        tmpData[k].z = pop[i].data[j].z;
                        tmpData[k].r = pop[i].data[j].r;
                        k++;
                    }
                }
                else {
                    tmpData[k].x = pop[i].data[j].x;
                    tmpData[k].y = pop[i].data[j].y;
                    tmpData[k].z = pop[i].data[j].z;
                    tmpData[k].r = pop[i].data[j].r;
                    k++;
                }
            }
            if (pop[i].nrObjects-nr_in_list>0){
                free (pop[i].data);
                pop[i].data = tmpData;
                pop[i].nrObjects = pop[i].nrObjects-nr_in_list;
            }
            else {
                free (tmpData);
            }
        }
        free (list);
    }
    return TRUE;
}

#endif /* __evobintom3D_h__ */
