#include			<time.h>



/**
***
***	MACROS & CONSTANTS
***
**/

#define bcopy(p1, p2, size) memcpy(p2, p1, size)
#define bzero(p1, size) memset(p1, 0, size)
#define read_random() rand()
/**
*** The logical TRUE value.
**/
#define			TRUE		(1 == 1)
/**
*** The logical FALSE value.
**/
#define			FALSE		!TRUE
/**
*** The smallest positive number. Any number less than EPS in magnitude
*** will be considered to be 0.
**/
#define			EPS		1E-6
/**
*** The smallest allowed radius. Used when searching a start configuration
*** or generating original projections.
**/
#define			MIN_RADIUS	5.0
/**
*** Default minimal radius to look for. Used when randomly adding
*** circles and when checking configuration conditions. Should be
*** greater than or equal to EPS.
**/
#define			MIN_SEARCH_RADIUS	5.0

/**
*** Default picture width (=height) in pixels. It equals the number of
*** data values of every projection.
**/
#define			PIC_SIZE	100

/**
*** File storing pre-generated random numbers.
**/
/* #define			RANDOM_FILE	"rand.t" */
/**
*** Number of pre-generated random numbers.
**/
/* #define			RANDOM_NUM	1000000 */

/**
*** Name of PGM file showing the original (ideal destination) configuration.
**/
#define			CONF_ORIGIN	"origin.pgm"

/**
*** Name of text file containing input (original) projections in DIRECT
*** format.
**/
#define			DIRECT_PROJ	"proj.txt"

/**
*** If set, these bits indicate actions to be done during exporting
*** projections and configurations in DIRECT format. Refer to structure
*** DIRECT_par.
**/
/**
***	No special actions at all. (Only fields i_input, circle_num and
***	fname are active.)
**/
#define			WRDIRECT_NONE		0
/**
***	Store a binary image representation of the configuration.
***	(Field in_config is active.)
**/
#define			WRDIRECT_IMAGE		1
/**
***	Store duration of reconstruction. (Field duration is active.)
**/
#define			WRDIRECT_DURATION	2
/**
***	Store name of input file. (Field in_fname is active.)
**/
#define			WRDIRECT_INPUT		4

/**
*** Error constants.
**/
/**
***	Normal termination.
**/
#define			ERR_OK			0
/**
***	Output log file could not be opened.
**/
#define			ERR_OPEN_O_LOG		1
/**
***	Parameter error.
**/
#define			ERR_PARAM		2
/**
***	File containing manually created circle parameters (i.e. the configuration)
***	could not be opened.
**/
#define			ERR_OPEN_PARAM_FILE	3
/**
***	Initial configuration violates conditions.
**/
#define			ERR_BAD_CONFIG		4
/**
***	Parameter min_search_radius is greater than the maximal allowed radius.
***	May occur when randomly adding circles to the configuration.
**/
#define			ERR_BAD_RADIUS		5
/**
***	File containing input projections could not be opened.
**/
#define			ERR_OPEN_PROJ_FILE	6
/**
***	Infinite loop encountered. May occur when randomly adding circles to the configuration.
**/
#define			ERR_INF_LOOP		7
/**
***	File containing randomly generated circle parameters (i.e. the configuration)
***	could not be saved.
**/
#define			ERR_SAVE_PARAM_FILE	8
/**
***	File containing input projections could not be saved.
**/
#define			ERR_SAVE_PROJ_FILE	9



/**
***
***	TYPE DEFINITIONS
***
**/

// Structure for parameters of involving cylinder and inner circles
typedef struct {
			// Number of circles stored in structure including the
			// two circles of involving cylinder.
			int	circle_num;
			// Params of circles (X, Y, Radius). The first six
			// elements are the params of the outer & inner circle
			// of involving cylinder.
			double	*params;
                        // LAC (linear attenuation coeffitient) value for
                        // cylinder. Should be positive.
                        double	lac_cyl;
                        // LAC value for regular circles. Should be positive.
                        double	lac_circ;
} par_s;

// Structure for projected data
typedef struct {
			// Angle of projection in degrees
			double	angle;
			// Lower bound of projection
			double	from;
			// Upper bound of projection
			double	to;
			// (x,y) coordinate of projected configuration
			double	center_X;
			double	center_Y;
			// Distance between two beams (reserved - normally set 1)
			double	distance;
			// Number of elements of the vector containing projection data
			int	data_num;
			// Vetctor that contains projected data
			double	*proj_data;
} projection;

// Structure for passing more than one projections to iterator
typedef struct {
			// Number of projections being passed to iterator
                        int		proj_num;
			// Pointer array. Each element points to a structure
			// of type projection.
			projection	**proj_arr;
} iter_input;

// Structure for passing parameters when exporting projections and
// configurations in DIRECT format.
typedef struct {
   			// Input projections
                        iter_input	*i_input;
                        // Input configuration of circles
                        par_s		*in_config;
                        // Number of circles
                        int		circle_num;
                        // Duration of reconstruction in seconds
                        time_t		duration;
                        // Name of input file
                        char		*in_fname;
                        // Boolean bitfield specifying special actions
                        // taken during export
                        char		action;
                        // Name of destination file
                        char		*fname;
} DIRECT_par;



/**
***
***	GLOBAL VARIABLES
***
**/

// Size of squarish picture to be reconstructed in pixels. This equals
// the number of projecting beams.
int	pic_size = PIC_SIZE;
// Text file containing random values.
FILE	*rand_file;
// Minimal radius to look for. Used when randomly adding circles and
// when checking configuration conditions.
double	min_search_radius = MIN_SEARCH_RADIUS;
// Controls debug mode. If set, debug messages will be generated.
// Tons of lines of output can be produced, so be careful!
char	debug = FALSE;
// Controls quiet mode. If set, no messages are written to the standard output at all.
char	quiet_mode = FALSE;
// Log file containing output messages.
//FILE	*o_log_file;
// Log file containing error messages.
//FILE	*e_log_file;
// If set, messages are enabled during checking configuration conditions.
char	cond_chk_msg = TRUE;



/**
***
***	FUNCTIONS
***
**/

/**
*** 	Calculates the square of the parameter (integer version).
*** 	input:	number of type int whose square is to be calculated
*** 	output: result (long int)
**/

long int sqr_i( int 	x)
{
    	long int    y;
	
	y = x;
	return y * y;
}



/**
*** 	Calculates the square of the parameter (double version).
*** 	input:	number of type double whose square is to be calculated
*** 	output: result (double)
**/

double sqr_d( double 	x)
{
    	return x * x;
}



/**
***	Constructing projection of circles
***	input:	pointer to a 'projection' structure which will contain result
***		pointer to a 'par_s' array which contains params of circles
***		angle of rotation in degrees
***		lower bound of projection
***		upper bound of projection
***		X coordinate of the center of rotation
***		Y coordinate of the center of rotation
***		distance between equidistant projection beams
***	output:	-
**/

void proj(	projection	*result,
		par_s		*in_params,
		double		in_angle,
		double		in_from,
		double		in_to,
		double		in_center_x,
		double		in_center_y,
		double		in_distance)
{
	double		lower, upper;	// lower & upper bound of projection
	par_s		rot;		// rotated circle parameters
	int		i, index;
	double		j;
	double		rad;		// rotation angle in radians
        double		y;

	// Determining lower & upper bound of projection
	lower = in_from < in_to ? in_from : in_to;
	upper = in_from < in_to ? in_to : in_from;

	// Converting the rotation angle to radians
	rad = (-in_angle - 90.0) / 180 * M_PI;

	// Filling in the result structure
	result->angle	 	= in_angle;
	result->from  		= lower;
	result->to    		= upper;
	result->data_num 	= (int) ( (upper - lower + 1) / in_distance);
	result->distance	= in_distance;
        result->center_X	= in_center_x;
        result->center_Y	= in_center_y;

	// Filling in the 'rot' structure
	rot.circle_num = in_params->circle_num;
	rot.params = (double*) malloc( 3 * in_params->circle_num * sizeof(double) );
	bzero(rot.params, 3 * in_params->circle_num * sizeof(double));
        rot.lac_cyl = in_params->lac_cyl;
        rot.lac_circ = in_params->lac_circ;

	// Rotating the (x,y) circle parameters
	for (i = 0; i < 3 * in_params->circle_num; i += 3) {
		rot.params[i] = (in_params->params[i] - in_center_x) * cos(rad) - (in_params->params[i + 1] - in_center_y) *
                 sin(rad) + in_center_x;
		rot.params[i + 1] = (in_params->params[i + 1] - in_center_y) * cos(rad) + (in_params->params[i] - in_center_x) *
                 sin(rad) + in_center_y;
		rot.params[i + 2] = in_params->params[i + 2];
	}

	// Memory allocation for projected data & filling with 0
	result->proj_data = (double*) malloc(result->data_num * sizeof(double));
	bzero(result->proj_data, result->data_num * sizeof(double));

	// Part I.
	// Projecting circles only, where 'j' is the step variable
	for (i = 6; i < 3 * in_params->circle_num; i += 3) {
		// Calculating first step in 'j'
		if (rot.params[i] - rot.params[i + 2] > lower) {
			index = (int) ((rot.params[i] - rot.params[i + 2] - lower) / in_distance);
			j = lower + index * in_distance;
		} else {
			j = lower;
			index = 0;
		}

		// Projecting
		for (j; (j <= rot.params[i] + rot.params[i + 2]) && (j < upper); j += in_distance, index++) {
                   	y = sqr_d(rot.params[i + 2]) - sqr_d(rot.params[i] - j);
			if ( y > 0 ) result->proj_data[index] += 2 * rot.lac_circ * sqrt(y);
                }
	}

	// Part II.
	// Projecting involving cylinder and adding values acquired in Part I.
	if (rot.params[0] - rot.params[2] > lower) {
		index = (int) ((rot.params[0] - rot.params[2] - lower) / in_distance);
		j = lower + index * in_distance;
	} else {
		j = lower;
		index = 0;
	}

	// Adding outer circle to projection
	for (j; (j <= rot.params[0] + rot.params[2]) && (j < upper); j += in_distance, index++) {
		y = sqr_d(rot.params[2]) - sqr_d(rot.params[0] - j);
                if ( y > 0 ) result->proj_data[index] += 2 * rot.lac_cyl * sqrt(y);
        }

	if (rot.params[3] - rot.params[5] > lower) {
		index = (int) ((rot.params[3] - rot.params[5] - lower) / in_distance);
		j = lower + index * in_distance;
	} else {
		j = lower;
		index = 0;
	}

	// Subtracting inner circle from projection
	for (j; (j <= rot.params[3] + rot.params[5]) && (j < upper); j += in_distance, index++) {
		y = sqr_d(rot.params[5]) - sqr_d(rot.params[3] - j);
                if ( y > 0 ) result->proj_data[index] -= 2 * rot.lac_cyl * sqrt(y);
        }

	// Freeing unused memory
	free(rot.params);

	// Returning
	return;
}


/**
***	Constructing projections of circles
***	input:	pointer to a 'projection*' structure which will contain results
***		pointer to a 'par_s' array which contains params of circles
***		pointer to angles of rotation in degrees
***     number of projection angles
***		lower bound of projection
***		upper bound of projection
***		X coordinate of the center of rotation
***		Y coordinate of the center of rotation
***		distance between equidistant projection beams
***	output:	-
**/

void proj_fast(	projection **result,
		par_s		*in_params,
		double*		in_angle,
		int         angle_nr,
		double		in_from,
		double		in_to,
		double		in_center_x,
		double		in_center_y,
		double		in_distance)
{
	double  lower, upper; // lower & upper bound of projection
	par_s*  rot;          // rotated circle parameters
	int     i, k, index, *ind;
    double  j, *jm;
    double* rad;          // rotation angle in radians
    double  y;
    int flag;

	// Determining lower & upper bound of projection
	lower = in_from < in_to ? in_from : in_to;
	upper = in_from < in_to ? in_to : in_from;

    // Allocating memory for rotations and radians
    rot = malloc(angle_nr*sizeof(par_s));
    rad = malloc(angle_nr*sizeof(double));
    ind = malloc(angle_nr*sizeof(int));
    jm  = malloc(angle_nr*sizeof(double));

	for (k=0; k<angle_nr; k++)
	{
        // Converting the rotation angle to radians
        rad[k] = (-in_angle[k] - 90.0) / 180 * M_PI;

        // Filling in the result structure
        result[k]->angle = in_angle[k];
        result[k]->from  = lower;
        result[k]->to    = upper;
        result[k]->data_num = (int) ( (upper - lower + 1) / in_distance);
        result[k]->distance = in_distance;
        result[k]->center_X = in_center_x;
        result[k]->center_Y = in_center_y;

        // Filling in the 'rot' structure
	    rot[k].circle_num = in_params->circle_num;
	    rot[k].params = (double*) malloc( 3 * in_params->circle_num * sizeof(double) );
	    bzero(rot[k].params, 3 * in_params->circle_num * sizeof(double));
        rot[k].lac_cyl = in_params->lac_cyl;
        rot[k].lac_circ = in_params->lac_circ;

        // Rotating the (x,y) circle parameters
        for (i = 0; i < 3 * in_params->circle_num; i += 3) {
            rot[k].params[i] = (in_params->params[i] - in_center_x) * cos(rad[k]) - (in_params->params[i + 1] - in_center_y) *
                sin(rad[k]) + in_center_x;
            rot[k].params[i + 1] = (in_params->params[i + 1] - in_center_y) * cos(rad[k]) + (in_params->params[i] - in_center_x) *
                sin(rad[k]) + in_center_y;
            rot[k].params[i + 2] = in_params->params[i + 2];
        }

        // Memory allocation for projected data & filling with 0
        result[k]->proj_data = (double*) malloc(result[k]->data_num * sizeof(double));
        bzero(result[k]->proj_data, result[k]->data_num * sizeof(double));
    }

    // Part I.
    // Projecting circles only, where 'j' is the step variable
    for (i = 6; i < 3 * in_params->circle_num; i += 3) {
        // Calculating first step in 'j'
        for (k=0; k<angle_nr; k++)
        {
            if (rot[k].params[i] - rot[k].params[i + 2] > lower) {
                ind[k] = (int) ((rot[k].params[i] - rot[k].params[i + 2] - lower) / in_distance);
                jm[k] = lower + ind[k] * in_distance;
            } else {
                jm[k] = lower;
                ind[k] = 0;
            }
        }

        index = ind[0];
        j     = jm[0];
        flag  = 1;
        for (k=1; k<angle_nr; k++)
        {
            if (j != jm[k] || index != ind[k])
               flag = 0;
        }
        
        if (flag == 1)
        {
            // Projecting
            for (j; (j <= rot[0].params[i] + rot[0].params[i + 2]) && (j < upper); j += in_distance, index++) {
                y = sqr_d(rot[0].params[i + 2]) - sqr_d(rot[0].params[i] - j);
                if ( y > 0 )
                {
                    for (k=0; k<angle_nr; k++)
                    {
                        result[k]->proj_data[index] += 2 * rot[k].lac_circ * sqrt(y);
                    }
                }
            }
        }
        else
        {
            // Projecting
            for (k=0; k<angle_nr; k++)
            {
                for (jm[k]; (jm[k] <= rot[k].params[i] + rot[k].params[i + 2]) && (jm[k] < upper); jm[k] += in_distance, ind[k]++) {
                    y = sqr_d(rot[k].params[i + 2]) - sqr_d(rot[k].params[i] - jm[k]);
                    if ( y > 0 )
                        result[k]->proj_data[ind[k]] += 2 * rot[k].lac_circ * sqrt(y);
                }
            }
        }
    }

    // Part II.
    // Projecting involving cylinder and adding values acquired in Part I.
    if (rot[0].params[0] - rot[0].params[2] > lower) {
        index = (int) ((rot[0].params[0] - rot[0].params[2] - lower) / in_distance);
        j = lower + index * in_distance;
    } else {
        j = lower;
        index = 0;
    }

    // Adding outer circle to projection
    for (j; (j <= rot[0].params[0] + rot[0].params[2]) && (j < upper); j += in_distance, index++) {
        y = sqr_d(rot[0].params[2]) - sqr_d(rot[0].params[0] - j);
        if ( y > 0 ) {
            for (k=0; k<angle_nr; k++)
            {
                result[k]->proj_data[index] += 2 * rot[k].lac_cyl * sqrt(y);
            }
        }
    }

    if (rot[0].params[3] - rot[0].params[5] > lower) {
        index = (int) ((rot[0].params[3] - rot[0].params[5] - lower) / in_distance);
        j = lower + index * in_distance;
    } else {
        j = lower;
        index = 0;
    }

    // Subtracting inner circle from projection
    for (j; (j <= rot[0].params[3] + rot[0].params[5]) && (j < upper); j += in_distance, index++) {
        y = sqr_d(rot[0].params[5]) - sqr_d(rot[0].params[3] - j);
        if ( y > 0 )
        {
            for (k=0; k<angle_nr; k++)
            {
                result[k]->proj_data[index] -= 2 * rot[k].lac_cyl * sqrt(y);
            }
        }
    }

    // Freeing unused memory
    for (k=0; k<angle_nr; k++) {
        free(rot[k].params);
    }
    free(rot);
    free(rad);
    free(jm);
    free(ind);

    // Returning
    return;
}

/**
***	Returns the greater of the arguments (double version).
***	input:	first argument (double)
***		second argument (double)
***	output:	the maximum value (double)
**/

double max_d(	double x,
   		double y)
{
   	return x < y ? y : x;
}



/**
***	Returns the greater of the arguments (int version).
***	input:	first argument (int)
***		second argument (int)
***	output:	the maximum value (int)
**/

int max_i(	int x,
   		int y)
{
   	return x < y ? y : x;
}



/**
***	Returns the less of the arguments (double version).
***	input:	first argument (double)
***		second argument (double)
***	output:	the minimum value (double)
**/

double min_d(	double x,
   		double y)
{
   	return x < y ? x : y;
}



/**
***	Returns the less of the arguments (int version).
***	input:	first argument (int)
***		second argument (int)
***	output:	the minimum value (int)
**/

int min_i(	int x,
   		int y)
{
   	return x < y ? x : y;
}



/**
***	Function that checks whether a given set of circles is disjoint or not.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***	output:	1,  if circles are disjoint
***		-1, if cylinder is broken
***		-2, if some of the circles have disappeared
***		-3, if circles are not disjoint (wrong center and radius)
***		-4, if circles are not disjoint (wrong radius)
***		-5, if some of the circles crashed into the cylinder (wrong center and radius)
***		-6, if some of the circles crashed into the cylinder (wrong radius)
**/

int cond_check(par_s	*in_params)
{
        double		radius_sum, radius_diff, max_radius;
	double	        center_dist;
	int		i, j;

	// Checking method for all circles:

        // Checks if the cylinder is correct
        if ((in_params->params[0] - in_params->params[2] < 0.0) ||
            (in_params->params[1] - in_params->params[2] < 0.0) ||
            (in_params->params[0] + in_params->params[2] >= pic_size) ||
            (in_params->params[1] + in_params->params[2] >= pic_size)) {
/*           if (debug && cond_chk_msg) {
              fprintf(o_log_file, "CONDITION CHECKER: Broken cylinder.\n");
           }*/
           return -1;
        }
	center_dist = sqrt(sqr_d(in_params->params[0] - in_params->params[3])\
	 + sqr_d(in_params->params[1] - in_params->params[4]));
        if ( center_dist + in_params->params[5] >= in_params->params[2] ) {
/*           if (debug && cond_chk_msg) {
              fprintf(o_log_file, "CONDITION CHECKER: Broken cylinder.\n");
           }*/
           return -1;
        }

	// Checks for disappearing circles
	for (i = 0; i < in_params->circle_num * 3; i += 3 ) {
		// Returning -2 if a circle has disappeared
//		if ( in_params->params[i + 2] < min_search_radius ) {
		if ( in_params->params[i + 2] - min_search_radius < -EPS ) {
/*                   if (debug && cond_chk_msg) {
                      fprintf(o_log_file, "CONDITION CHECKER: Circle disappeared.\n");
                   }*/
                   return -2;
		}
	}

	// Checks whether all the circle pairs in the set are disjoint or not
	for (i = 6; i < (in_params->circle_num - 1) * 3; i +=3 ) {
	    for (j = i + 3; j < in_params->circle_num * 3; j +=3 ) {
		radius_sum  = in_params->params[j + 2] + in_params->params[i + 2];
                max_radius = max_d(in_params->params[i + 2], in_params->params[j + 2]);
		center_dist = sqrt(sqr_d(in_params->params[j] - in_params->params[i])\
				 + sqr_d(in_params->params[j + 1] - in_params->params[i + 1]));

		// Returning -3 or -4 if circles are not disjoint
//		if ( center_dist < radius_sum ) {
		if ( center_dist - radius_sum < -EPS ) {
/*                   if (debug && cond_chk_msg) {
                      fprintf(o_log_file, "CONDITION CHECKER: Circles are not disjoint ");
                   }*/
                   if ( center_dist <= max_radius ) {
/*                      if (debug && cond_chk_msg) {
                         fprintf(o_log_file, "(wrong center and radius).\n");
                      }*/
                      return -3;
                   }
                   else {
/*                      if (debug && cond_chk_msg) {
                         fprintf(o_log_file, "(wrong radius).\n");
                      }*/
                      return -4;
                   }
		}
	    }
	}

	// Checking if any circle crash into the cylinder
	for (i = 6; i < in_params->circle_num * 3; i += 3 ) {
           	radius_diff = in_params->params[5] - in_params->params[i + 2];
		center_dist = sqrt(sqr_d(in_params->params[3] - in_params->params[i])\
			+ sqr_d(in_params->params[4] - in_params->params[i + 1]));

		// Returning -5 or -6 if a circle has crashed into the cylinder
//		if ( center_dist > radius_diff ) {
		if ( center_dist - radius_diff > EPS ) {
/*                   if (debug && cond_chk_msg) {
                      fprintf(o_log_file, "CONDITION CHECKER: Circle crashed into the cylinder ");
                   }*/
                   if (center_dist >= in_params->params[5] ) {
/*                      if (debug && cond_chk_msg) {
                         fprintf(o_log_file, "(wrong center and radius).\n");
                      }*/
                      return -5;
                   }
		   else {
/*                      if (debug && cond_chk_msg) {
                         fprintf(o_log_file, "(wrong radius).\n");
                      }*/
                      return -6;
                   }
		}
	}

	// Returning 1 if circles are disjoint
	return 1;
}

/**
***	Function that checks whether a given set of circles is disjoint or not.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***         number of the investigated circle
***	output:	1,  if circles are disjoint
***		-1, if cylinder is broken
***		-2, if some of the circles have disappeared
***		-3, if circles are not disjoint (wrong center and radius)
***		-4, if circles are not disjoint (wrong radius)
***		-5, if some of the circles crashed into the cylinder (wrong center and radius)
***		-6, if some of the circles crashed into the cylinder (wrong radius)
**/

int cond_check_fast(par_s	*in_params, int circle, double temp_x, double temp_y, double temp_r)
{
    double radius_sum, radius_diff, max_radius;
    double center_dist;
    int    i;
    if (circle == 0)
    {
        // Checks if the cylinder is correct
        if ((temp_x - temp_r < 0.0) ||
            (temp_y - temp_r < 0.0) ||
            (temp_x + temp_r >= pic_size) ||
            (temp_y + temp_r >= pic_size)) {
           return -1;
        }
    }
    else if (circle == 1)
    {
        center_dist = sqrt(sqr_d(in_params->params[0] - temp_x)\
	        + sqr_d(in_params->params[1] - temp_y));
        if ( center_dist + temp_r >= in_params->params[2] ) {
            return -1;
        }
    }
    else
    {
	    // Checks for disappearing circles
        // Returning -2 if a circle has disappeared
        if (temp_r - min_search_radius < -EPS ) {
            return -2;
        }

    	// Checks whether all the circle pairs in the set are disjoint or not
	    for (i = 6; i < (in_params->circle_num - 1) * 3; i +=3 ) {
            if (i == (circle*3))
            {
                continue;
            }
    		radius_sum  = temp_r + in_params->params[i + 2];
            max_radius = max_d(in_params->params[i + 2], temp_r);
		    center_dist = sqrt(sqr_d(temp_x - in_params->params[i])\
			    	 + sqr_d(temp_y - in_params->params[i + 1]));

    		// Returning -3 or -4 if circles are not disjoint
            if ( center_dist - radius_sum < -EPS ) {
                if ( center_dist <= max_radius ) {
                    return -3;
                }
                else {
                    return -4;
                }
            }
        }
    	// Checking if circle crash into the cylinder
        radius_diff = in_params->params[5] - temp_r;
        center_dist = sqrt(sqr_d(in_params->params[3] - temp_x)\
            + sqr_d(in_params->params[4] - temp_y));
        // Returning -5 or -6 if a circle has crashed into the cylinder
        if ( center_dist - radius_diff > EPS ) {
            if (center_dist >= in_params->params[5] ) {
                return -5;
            }
            else {
                return -6;
            }
		}
	}

	// Returning 1 if circles are disjoint
	return 1;
}

/**
***	Decides whether a single circle of given radius can be safely
***	added to the configuration.
***	input:	input configuration (should contain at least 2 inner circles)
***		radius in question
***	output:	TRUE, if radius is safe
***		FALSE, otherwise
**/

char is_radius_safe(par_s	*in_params,
   		    double	r)
{
   	double		min_search_radius_save;
        par_s		p;
        char		safe = FALSE;
        int		circle_num;
        int		i, j;
        double		min_r, max_r_1, max_r_2;
        double		d, x1, y1, r1, x2, y2, r2, ratio;
        double		mx, my, b, c, sinalpha, cosalpha, t1, t2;
        
        
        // Cloning input configuration
        circle_num = in_params->circle_num;
        p.circle_num = circle_num + 1;
        p.params = malloc(p.circle_num * 3 * sizeof(double));
        bcopy(in_params->params, p.params, circle_num * 3 * sizeof(double));
        
        // Back up parameter and set it to a safe value
        min_search_radius_save = min_search_radius;
        min_search_radius = r;
        for (i = 2; i < circle_num; i++)
           min_search_radius = min_d(min_search_radius, p.params[i * 3 + 2]);
        
        // Setting radius of new circle
        p.params[circle_num * 3 + 2] = r;
        
        // Get parameters of the inner circle of the cylinder
        x2 = p.params[3];
        y2 = p.params[4];
        r2 = p.params[5];

        // Examining pairs of the inner circle of the cylinder
        // and a regular circle
        for (i = 2; (i < circle_num) && !safe; i++) {
           // Get parameters of the regular circle
           x1 = p.params[i * 3];
           y1 = p.params[i * 3 + 1];
           r1 = p.params[i * 3 + 2];

           // Calculate distance of centers and maximal allowed radius
           d = sqrt(sqr_d(x1 - x2) + sqr_d(y1 - y2));
           max_r_1 = (r2 - r1 - d) / 2.0;
           max_r_2 = (r2 - r1 + d) / 2.0;
           
           // Skip this circle if the centers coincide
           if (d > EPS) {
              // Calculate centers of the tangent circles contacting
              // the inner circle of the cylinder
              sinalpha = (y2 - y1) / d;
              cosalpha = (x2 - x1) / d;
              b = 2 * cosalpha * (x1 - x2) + 2 * sinalpha * (y1 - y2);
              c = sqr_d(x1) + sqr_d(x2) - 2 * x1 * x2 +
                  sqr_d(y1) + sqr_d(y2) - 2 * y1 * y2 - sqr_d(r2);
              t1 = (-b + sqrt(sqr_d(b) - 4 * c)) / 2.0;
              t2 = (-b - sqrt(sqr_d(b) - 4 * c)) / 2.0;
              
              // Try both common tangent circles
              if (r <= max_r_1) {
                 p.params[circle_num * 3] = x1 + (t2 + r) * cosalpha;
                 p.params[circle_num * 3 + 1] = y1 + (t2 + r) * sinalpha;
                 if ( cond_check(&p) == 1 ) safe = TRUE;
              } else if (r <= max_r_2) {
                 p.params[circle_num * 3] = x1 + (t1 - r) * cosalpha;
                 p.params[circle_num * 3 + 1] = y1 + (t1 - r) * sinalpha;
                 if ( cond_check(&p) == 1 ) safe = TRUE;
              }
           }
        }

        // Examining pairs of regular circles
        for (i = 2; (i < circle_num - 1) && !safe; i++)
           for (j = i + 1; (j < circle_num) && !safe; j++) {
              // Get parameters of the two circles
              x1 = p.params[i * 3];
              y1 = p.params[i * 3 + 1];
              r1 = p.params[i * 3 + 2];
              x2 = p.params[j * 3];
              y2 = p.params[j * 3 + 1];
              r2 = p.params[j * 3 + 2];
              
              // Calculate distance of centers and minimal allowed radius
              d = sqrt(sqr_d(x1 - x2) + sqr_d(y1 - y2));
              min_r = (d - r1 - r2) / 2.0;

              if ( r < min_r ) {
                 // Try two tangent circles of radius r with their center
                 // lieing on segment (x1,y1)-(x2,y2)
                 ratio = (r1 + r) / (d - (r1 + r));
                 p.params[circle_num * 3] = (x1 + ratio * x2) /
                                            (1.0 + ratio);
                 p.params[circle_num * 3 + 1] = (y1 + ratio * y2) /
                                                (1.0 + ratio);
                 if ( cond_check(&p) == 1 ) safe = TRUE;
                 else {
                     ratio = (d - (r2 + r)) / (r2 + r);
                     p.params[circle_num * 3] = (x1 + ratio * x2) /
                                                (1.0 + ratio);
                     p.params[circle_num * 3 + 1] = (y1 + ratio * y2) /
                                                    (1.0 + ratio);
                     if ( cond_check(&p) == 1 ) safe = TRUE;
                 }
              } else {
                 // Calculate centers of the common tangent circles
                 b = (sqr_d(r1 + r) - sqr_d(r2 + r) + sqr_d(d)) / (2.0 * d);
                 ratio = b / (d - b);
                 mx = (x1 + ratio * x2) / (1.0 + ratio);
                 my = (y1 + ratio * y2) / (1.0 + ratio);
                 sinalpha = (x1 - x2) / d;
                 cosalpha = (y2 - y1) / d;
                 b = 2 * cosalpha * (mx - x1) + 2 * sinalpha * (my - y1);
                 c = sqr_d(mx) + sqr_d(x1) - 2 * mx * x1 +
                     sqr_d(my) + sqr_d(y1) - 2 * my * y1 - sqr_d(r1 + r);
                 t1 = (-b + sqrt(sqr_d(b) - 4 * c)) / 2.0;
                 t2 = (-b - sqrt(sqr_d(b) - 4 * c)) / 2.0;
                 
                 // Try both common tangent circles
                 p.params[circle_num * 3] = mx + t1 * cosalpha;
                 p.params[circle_num * 3 + 1] = my + t1 * sinalpha;
                 b = sqrt(sqr_d(p.params[circle_num * 3] - x1) + 
                          sqr_d(p.params[circle_num * 3 + 1] - y1)) - r1;
                 c = sqrt(sqr_d(p.params[circle_num * 3] - x2) + 
                          sqr_d(p.params[circle_num * 3 + 1] - y2)) - r2;
                 p.params[circle_num * 3 + 2] = min_d(b, c);
                 if ( cond_check(&p) == 1 ) safe = TRUE;
                 else {
                     p.params[circle_num * 3] = mx + t2 * cosalpha;
                     p.params[circle_num * 3 + 1] = my + t2 * sinalpha;
                     b = sqrt(sqr_d(p.params[circle_num * 3] - x1) + 
                              sqr_d(p.params[circle_num * 3 + 1] - y1)) - r1;
                     c = sqrt(sqr_d(p.params[circle_num * 3] - x2) + 
                              sqr_d(p.params[circle_num * 3 + 1] - y2)) - r2;
                     p.params[circle_num * 3 + 2] = min_d(b, c);
                     if ( cond_check(&p) == 1 ) safe = TRUE;
                 }
              }
           }
        
        // Restore parameter
        min_search_radius = min_search_radius_save;
        
        // Freeing allocated memory
        free(p.params);
        
        return safe;
}



/**
***	Calculates the maximal radius of a single circle that can be safely
***	added to the configuration.
***	input:	input configuration
***	output:	maximal radius
**/

double get_max_radius(par_s	*in_params)
{
   	double		a, b, x;
        double		delta = 1E-3;


        if ( in_params->circle_num < 4 ) {
           if ( in_params->circle_num == 2 ) a = in_params->params[5];
           else {
              a = in_params->params[5] - in_params->params[8] +
                  sqrt(sqr_d(in_params->params[3] - in_params->params[6]) +
                       sqr_d(in_params->params[4] - in_params->params[7]));
              a /= 2.0;
           }
        } else {
           // Find maximal radius by interval halving
           a = 0.0;
           b = in_params->params[5];
           while ( b - a >= delta ) {
              x = (a + b) / 2.0;
              if ( is_radius_safe(in_params, x) ) a = x;
              else b = x;
           }
        }
        
        return a;
}



/**
***	Randomly adds some circles to the given configuration.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***		how many circles to add
***		minimal radius to use
***	output:	-
**/

int add_circle(par_s *in_params, int num, double min_r) {
    int i, j;
    int index, index1, index2;
    int random;
    double unif;
    double x1, y1, x2, y2, max_r;
    int iter;

    if ( num < 1 ) return -2;

    index = in_params->circle_num;
        
    // Initialisating radii
    for (index1 = index, i = 0; i < num; index1++, i++)
        in_params->params[index1 * 3 + 2] = min_r;

    // Setting centers
    x1 = in_params->params[3] - in_params->params[5];
    y1 = in_params->params[4] - in_params->params[5];
    x2 = in_params->params[3] + in_params->params[5];
    y2 = in_params->params[4] + in_params->params[5];
    for (index1 = index, i = 0; i < num; index1++, i++) {
        max_r = get_max_radius(in_params);
        if ( min_r > max_r + EPS ) {
            return -1;
        }

        in_params->circle_num++;
        iter = 0;
        do {
            iter++;
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3] = x1 + unif * (x2 - x1);
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3 + 1] = y1 + unif * (y2 - y1);
            j = cond_check_fast(in_params, index1, in_params->params[index1 * 3], in_params->params[index1 * 3+1], in_params->params[index1 * 3+2]);
        } while ( (j != 1) && (iter < 1000) );

        if ( iter == 1000 ) {
             return -1;
        }
    }
        
    // Generating radii
    for (index1 = index, i = 0; i < num; index1++, i++) {
        // Calculating maximal allowed radius
        x1 = in_params->params[index1 * 3];
        y1 = in_params->params[index1 * 3 + 1];
        x2 = in_params->params[3];
        y2 = in_params->params[4];
        max_r = in_params->params[5] - sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) );
        index2 = 2;
        if ( index2 != index ) {
            for (index2; index2 < index1; index2++) {
                x2 = in_params->params[index2 * 3];
                y2 = in_params->params[index2 * 3 + 1];
                max_r = min_d(max_r, sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) ) -
                      in_params->params[index2 * 3 + 2]);
            }
            for (++index2; index2 < index + num; index2++) {
                x2 = in_params->params[index2 * 3];
                y2 = in_params->params[index2 * 3 + 1];
                max_r = min_d(max_r, sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) ) -
                    in_params->params[index2 * 3 + 2]);
            }
        }

        // Generating radius randomly
        do {
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3 + 2] = min_r + unif * (max_r - min_r);
            j = cond_check(in_params);
//            j = cond_check_fast(in_params, index1, in_params->params[index1 * 3], in_params->params[index1 * 3+1], in_params->params[index1 * 3+2]);
        } while ( j != 1 );
    }

	return 0;
}

/**
***	Randomly adds some circles to the given configuration.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***		how many circles to add
***		minimal radius to use
***	output:	-
**/

int add_circle_fast(par_s *in_params, int num, double min_r) {
    int i, j;
    int index, index1, index2;
    int random;
    double unif;
    double x1, y1, x2, y2, max_r;
    int iter;
    int max_iter = 100;

    if ( num < 1 ) return -2;

    index = in_params->circle_num;
        
    // Initialisating radii
    for (index1 = index, i = 0; i < num; index1++, i++)
        in_params->params[index1 * 3 + 2] = min_r;

    // Setting centers
    x1 = in_params->params[3] - in_params->params[5];
    y1 = in_params->params[4] - in_params->params[5];
    x2 = in_params->params[3] + in_params->params[5];
    y2 = in_params->params[4] + in_params->params[5];
    for (index1 = index, i = 0; i < num; index1++, i++) {
        max_r = get_max_radius(in_params);
        if ( min_r > max_r + EPS ) {
            return -1;
        }

        in_params->circle_num++;
        iter = 0;
        do {
            iter++;
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3] = x1 + unif * (x2 - x1);
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3 + 1] = y1 + unif * (y2 - y1);
            j = cond_check_fast(in_params, index1, in_params->params[index1 * 3], in_params->params[index1 * 3+1], in_params->params[index1 * 3+2]);
        } while ( (j != 1) && (iter < max_iter) );

        if ( iter == max_iter ) {
             return -1;
        }
    }
        
    // Generating radii
    for (index1 = index, i = 0; i < num; index1++, i++) {
        // Calculating maximal allowed radius
        x1 = in_params->params[index1 * 3];
        y1 = in_params->params[index1 * 3 + 1];
        x2 = in_params->params[3];
        y2 = in_params->params[4];
        max_r = in_params->params[5] - sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) );
        index2 = 2;
        if ( index2 != index ) {
            for (index2; index2 < index1; index2++) {
                x2 = in_params->params[index2 * 3];
                y2 = in_params->params[index2 * 3 + 1];
                max_r = min_d(max_r, sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) ) -
                      in_params->params[index2 * 3 + 2]);
            }
            for (++index2; index2 < index + num; index2++) {
                x2 = in_params->params[index2 * 3];
                y2 = in_params->params[index2 * 3 + 1];
                max_r = min_d(max_r, sqrt( sqr_d(x1 - x2) + sqr_d(y1 - y2) ) -
                    in_params->params[index2 * 3 + 2]);
            }
        }

        // Generating radius randomly
        do {
            random = read_random();
            unif = random / (double) RAND_MAX;
            in_params->params[index1 * 3 + 2] = min_r + unif * (max_r - min_r);
            j = cond_check(in_params);
//            j = cond_check_fast(in_params, index1, in_params->params[index1 * 3], in_params->params[index1 * 3+1], in_params->params[index1 * 3+2]);
        } while ( j != 1 );
    }

	return 0;
}

/**
***	Renders reconstructed configuration to memory.
***	Origin is put at the bottom-left corner of the image.
***	input:	pointer to structure containing reconstructed configuration
***		X dimension of picture (for exportation in PGM format)
***		Y dimension of picture (for exportation in PGM format)
***		maximal intensity
***	output:	address of array storing image
**/

unsigned char *render_conf(par_s		*in_config,
   			   int			Xsize,
                           int			Ysize,
                           unsigned char	maxint)
{
    unsigned char	*pic;
    int			i, j, c;


    // Memory allocation for the reconstructed binary picture and
    // filling by zeros
    pic = (unsigned char *) malloc(Xsize * Ysize * sizeof(unsigned char));
    bzero(pic, Xsize * Ysize * sizeof(unsigned char));
    
    // Examining each pixel
    // If a pixel is a circle (cylinder) element -> maximal intensity
    // If a pixel is not a circle (cylinder) element -> zero intensity
    for ( j = 0; j < Ysize; j++ )
	for ( i = 0; i < Xsize; i++ ) {
	    if  (((sqr_d(i - in_config->params[0]) + sqr_d(j - in_config->params[1]) - sqr_d(in_config->params[2]) <= 0) &&
		(sqr_d(i - in_config->params[3]) + sqr_d(j - in_config->params[4]) - sqr_d(in_config->params[5]) > 0)))\
		pic[(Ysize - 1 - j) * Xsize + i] = maxint;
	    for ( c = 6; c < 3 * in_config->circle_num; c+=3 )
		if ((sqr_d(i - in_config->params[c]) +\
		     sqr_d(j - in_config->params[c + 1]) -\
		     sqr_d(in_config->params[c + 2]) <= 0 ))
		    pic[(Ysize - 1 - j) * Xsize + i] = maxint;
	    }

    return pic;
}



/**
***	Function exports a configuration in PGM format.
***	Origin is put at the bottom-left corner of the image.
***	input:	pointer to configuration
***		X dimension of picture (for exportation in PGM format)
***		Y dimension of picture (for exportation in PGM format)
***		maximal intensity
***		name of exported file
***	output:	1 on successful exporting
***		-1 on occuring error(s)
**/

int write_PGM(	par_s 		*in_config,
		int		Xsize,
		int	        Ysize,
		unsigned char	maxint,
		char	        *fname)
{
    FILE		*out_file;
    unsigned char	*pic;


    // Opening output file. Returning -1 on error
    if ( !(out_file = fopen(fname, "w")) ) return -1;

    // Writing PGM header (<P5> <xsize> <ysize> <maximal intensity><CR/LF>)
    fprintf(out_file, "P5 %d %d %d\n", Xsize, Ysize, maxint);

    // Rendering image
    pic = render_conf(in_config, Xsize, Ysize, maxint);

    // Writing raw binary picture into output file
    fwrite(pic, 1, Xsize * Ysize * sizeof(unsigned char), out_file);

    // Freeing allocated memory
    free(pic);

    // Closing output file. Returning -1 on error
    if ( fclose(out_file) ) return -1;

    // Normal function termination
    return 1;
}



/**
***	Generating random values into file
***	Input:	name of file into random values are stored
***		number of random values to be generated
***	Output:	-1 on unsuccessful operation
***		i  on successful operation
**/		
/* most ez nem kell
int gen_random(char *file_name, int random_num)
{
    FILE	*f;
    int		i;

    if ( !(f = fopen(file_name, "w")) ) return -1;

    // Reading parameters
    for (i = 0; i < random_num; i++)
	fprintf(f, "%10d\n", rand());

    fclose(f);
    return 1;
}
*/


/**
***	Dumps projection data to file.
***	Input:	pointer to a projection
***		pointer to an opened FILE structure
***	Output:	-1 on unsuccessful operation
***		1  on successful operation
**/

int dump_proj(projection *p, FILE *f)
{
    int	i;

    for ( i = 0; i < p->data_num; i++ )
       fprintf(f, "%f\n", p->proj_data[i]);
    fprintf(f, "\n");
    return 1;
}



/**
***
***	Exports given set of projections and optionally a configuration
***	in DIRECT format.
***	Input:	address of structure containing parameters
***	Output:	-1 on unsuccessful operation
***		1  on successful operation
***
**/

int write_DIRECT(DIRECT_par	*pars)
{
    FILE		*f;
    unsigned char	*pic;
    int			i, j, k;
    char		write_pic, write_duration, write_input;
    

    if ( !(f = fopen(pars->fname, "w")) ) return -1;

    write_pic = pars->action & WRDIRECT_IMAGE;
    write_duration = pars->action & WRDIRECT_DURATION;
    write_input = pars->action & WRDIRECT_INPUT;

    fprintf(f, "DIMENSIONS 2\n\n");
    
    if ( write_pic ) {
       pic = render_conf(pars->in_config, pic_size, pic_size, 1);
       fprintf(f, "IMAGE %d %d", pic_size, pic_size);
       j = 0;
       for (i = 0; i < sqr_i(pic_size); i++ ) {
          if ( !(j++) ) fprintf(f, "\n");
          fprintf(f, "  %d", pic[i]);
          if ( j == 10) j = 0;
       }
       fprintf(f, "\n\n");
       free(pic);
    }

    fprintf(f, "NRPROJECTIONS %d\n\n", pars->i_input->proj_num);

    for (i = 0; i < pars->i_input->proj_num; i++ ) {
       fprintf(f, "PROJECTION %d ANGLE %g",
               pars->i_input->proj_arr[i]->data_num,
               pars->i_input->proj_arr[i]->angle);
       k = 0;
       for (j = 0; j < pars->i_input->proj_arr[i]->data_num; j++ ) {
          if ( !(k++) ) fprintf(f, "\n");
          fprintf(f, "  %g", pars->i_input->proj_arr[i]->proj_data[j]);
          if ( k == 6) k = 0;
       }
       fprintf(f, "\n\n");
    }

    fprintf(f, "FILE %s\n\n", pars->fname);

    if ( write_input ) fprintf(f, "INPUT %s\n\n", pars->in_fname);

    if ( write_pic) fprintf(f, "NRSOLUTIONS 1\n\n");

    fprintf(f, "NRCIRCLES %d\n\n", pars->circle_num);

    if ( write_duration ) fprintf(f, "DURATION %d\n\n", pars->duration);

    fclose(f);
    
    return 1;
}



/**
***
***	Converts a string to upper-case in-place.
***	Input:	string to be converted
***	Output: -
***
**/

void strupcase(char* s)
{
   if ( !s ) return;

   while ( s[0] ) {
      s[0] = toupper(s[0]);
      s++;
   }
   
   return;
}



/**
***	Saves circle parameters of configuration to file.
***	Input:	name of file
***		address of structure containing parameters
***	Output:	1 if operation was successful
***		-1 on unsuccessful operation
**/

int save_params(char	*file_name,
		par_s	*p)
{
    FILE	*f;
    int		i;

    if ( !(f = fopen(file_name, "a+")) ) return -1;

    fprintf(f, "%d\n", p->circle_num);
    for ( i = 0; i < p->circle_num; i++ ) 
	fprintf(f, "%g %g %g\n", p->params[3 * i], p->params[3 * i + 1], p->params[3 * i + 2]);

    fclose(f);
    return 1;
}
