#include			<math.h>
#include			<stdlib.h>
#include			<string.h>
#include			<stdio.h>
#include			<unistd.h>
#include			"specrec.h"



/**
***
***	MACROS & CONSTANTS
***
**/

/**
***
*** this macro generates a random number between the two given integers
***
**/
#define getrandom( min, max ) ((rand() % (int)(((max)+1) - (min))) + (min))

/**
*** Default values of various parameters.
**/
/**
*** Number of projections used.
**/
#define			PROJ_NUM	4
/**
*** Angles of rotations in degrees specifying projecting directions.
**/
#define			PROJ_ANGLE	{0.0, 45.0, 90.0, 135.0}
/**
*** If TRUE, simulated noise will be added to input projections.
**/
#define			USE_NOISE	TRUE
/**
*** Amount of noise in percents to be added to the projections. Let
*** R be a random real number of the interval [0.0; 1.0] (both included),
*** M be the maximal value of a given projection. Then, for every P
*** projection value, the following N is calculated:
***
***    N = (R - 0.5) * M * NOISE_PERC / 50.0 .
***
*** Finally, new projection value P' is calculated as
***
***    P' = MAX(P + N, 0.0),
***
*** where MAX(x, y) returns the greater of the arguments.
***
**/
#define			NOISE_PERC	15
/**
*** LAC (linear attenuation coeffitient) value for cylinder. Should be
*** positive.
**/
#define			LAC_CYL		1.0
/**
*** LAC value for regular circles. Should be positive.
**/
#define			LAC_CIRC	1.0

/**
*** Log file containing output messages.
**/
#define			OUTPUT_LOG	"projgen_o.log"

/**
*** Log file containing error messages.
**/
#define			ERROR_LOG	"projgen_e.log"

/**
*** Name of text file containing original configuration and projections
*** in DIRECT format.
**/
#define			DIRECT_ORIGIN	"origin.txt"

/**
*** Name of initialisation file.
**/
#define			INIT_FILE	"init.t"

/**
*** Name of text file containing parameters for the manually created
*** original configuration.
**/
#define			IN_PARS_ORIGIN	"circles_o.t"

/**
*** Name of text file containing parameters for the randomly generated
*** original configuration.
**/
#define			IN_PARS_GEN	"circles_g.t"

/**
*** Name of text file containing noiseless projections.
**/
#define			FILTER_ORIGIN	"filter_o.t"
/**
*** Name of text file containing noisy projections.
**/
#define			FILTER_NOISY	"filter_n.t"

/**
*** Infinitive
**/
#define          INF             10000000.0

/**
***
***	TYPE DEFINITIONS
***
**/

// Structure for initial projection parameters.
typedef struct {
   			// Number of projections.
                        int		proj_num;
                        // Projecting directions in degrees.
                        double		*proj_angle;
                        // If TRUE, projections will be made noisy.
                        char		use_noise;
                        // Amount of noise in percents to be added.
                        double		noise_perc;
                        // LAC (linear attenuation coeffitient) value for
                        // cylinder. Should be positive.
                        double		lac_cyl;
                        // LAC value for regular circles. Should be positive.
                        double		lac_circ;
} init_s;



/**
***
***	GLOBAL VARIABLES
***
**/

int picture_number = 0;
int print_best = 0;       // boolean: print every best value configuration to file

time_t proj_time;
time_t cond_time;
time_t mut1_time, mut2_time, mut3_time, mut4_time;
time_t mrad_time;

/**
***
***	FUNCTIONS
***
**/

/**
***	Randomly adds cylinder to the given configuration.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***	output:	--
**/

void add_cylinder(par_s	*in_params)
{
   	int		i;
        int		random;
	double		unif;
        double		outer_start_max = 10.0;
        double		cyl_thick_min = 2.0, cyl_thick_max = 10.0;


        in_params->circle_num = 2;

        // Setting center
        in_params->params[0] = in_params->params[3] = pic_size / 2.0;
        in_params->params[1] = in_params->params[4] = pic_size / 2.0;
        
        // Generating radii
        random = read_random();
        unif = random / (double) RAND_MAX;
        in_params->params[2] = pic_size / 2.0 - unif * outer_start_max;
        random = read_random();
        unif = random / (double) RAND_MAX;
        in_params->params[5] = in_params->params[2] - cyl_thick_min -
                               unif * cyl_thick_max;

	return;
}



/**
***	Reads circle parameters of configuration from file.
***	Input:	name of file
***		address of structure containing parameters on return
***	Output:	1 if operation was successful
***		-1 on unsuccessful operation
**/

int read_params(char	*file_name,
		par_s	*p)
{
    FILE	*f;
    int		circle_num;
    int		i;
    float	x, y, r;

    if ( !(f = fopen(file_name, "r")) ) return -1;

    fscanf(f, "%d\n", &circle_num);
    p->circle_num = circle_num;
    p->params = malloc(3 * circle_num * sizeof(double));
    for ( i = 0; i < circle_num; i++ ) {
	fscanf(f, "%f %f %f\n", &x, &y, &r);
        p->params[3 * i] = x;
	p->params[3 * i + 1] = y;
	p->params[3 * i + 2] = r;
    }

    fclose(f);
    return 1;
}

int read_more_params(char	*file_name,
		par_s	**p, int* number)
{
    FILE	*f;
    int		circle_num;
    int		i, j;
    float	x, y, r;

    if ( !(f = fopen(file_name, "r")) ) return -1;
    j = 0;
    while (fscanf(f, "%d\n", &circle_num)!=EOF)
    {
        
        p[j] = malloc(sizeof(par_s));
        p[j]->circle_num = circle_num;
        p[j]->params = malloc(3 * circle_num * sizeof(double));
        for ( i = 0; i < circle_num; i++ ) {
           	fscanf(f, "%f %f %f\n", &x, &y, &r);
            p[j]->params[3 * i] = x;
            p[j]->params[3 * i + 1] = y;
	        p[j]->params[3 * i + 2] = r;
        }
        j++;
    }
    *number=j;

    fclose(f);
    return 1;
}



/**
***	Reads initial parameters from file, or sets them to default values.
***	Input:	name of file
***		structure containing initalisation parameters on return
***	Output:	-1, if parameter of keyword PROJ_NUM is wrong
***		1  on successful operation
**/

int read_init(char	*file_name,
              init_s	*init)
{
   	FILE		*f;
        char		*buff, *str_par, *str_val;
        double		proj_angle_default[] = PROJ_ANGLE;
        int		i;

    
        if ( !(f = fopen(INIT_FILE, "r")) ) {
                init->proj_num = PROJ_NUM;
                init->proj_angle = malloc(PROJ_NUM * sizeof(double));
                for (i = 0; i < PROJ_NUM; i++)
                   init->proj_angle[i] = proj_angle_default[i];
                init->use_noise = USE_NOISE;
                init->noise_perc = NOISE_PERC;
                init->lac_cyl = LAC_CYL;
                init->lac_circ = LAC_CIRC;

                return 1;
        } else {
                buff = malloc(256 * sizeof(char));
                str_par = malloc(256 * sizeof(char));
                str_val = malloc(256 * sizeof(char));
                init->proj_num = i = 0;
                init->proj_angle = NULL;
                init->use_noise = FALSE;
                init->noise_perc = 0.0;
                init->lac_cyl = LAC_CYL;
                init->lac_circ = LAC_CIRC;
                while ( !feof(f) ) {
                   buff[0] = '\0';
                   fgets(buff, 256, f);
                   buff[255] = '\0';
                   strupcase(buff);
                   str_par[0] = '\0';
                   str_val[0] = '\0';
                   sscanf(buff, " %[A-Za-z_] = %[.0-9eE+-]", str_par, str_val);
                   if ( strlen(str_par) ) {
                      if ( !strcmp("PROJ_NUM", str_par) ) {
                         init->proj_num = atoi(str_val);
                         if ( init->proj_num > 0)
                            init->proj_angle = malloc(init->proj_num * sizeof(double));
                      } else if ( !strcmp("PROJ_ANGLE", str_par) ) {
                         if ( i < init->proj_num )
                            init->proj_angle[i++] = strtod(str_val, NULL);
                      } else if ( !strcmp("NOISE_PERC", str_par) ) {
                         init->use_noise = TRUE;
                         init->noise_perc = strtod (str_val, NULL);
                      } else if ( !strcmp("PIC_SIZE", str_par) ) {
                         pic_size = atoi (str_val);
                      } else if ( !strcmp("LAC_CYL", str_par) ) {
                         init->lac_cyl = strtod (str_val, NULL);
                      } else if ( !strcmp("LAC_CIRC", str_par) ) {
                         init->lac_circ = strtod (str_val, NULL);
                      }
                   }
                }
		fclose(f);
                free(buff);
                free(str_par);
                free(str_val);

                return i != init->proj_num ? -1 : 1;
        }
}



/**
***	Adds uniformly distributed additive noise to projection.
***	Input:	pointer to a projection to be added noise
***		percentage of added noise
***	Outpu:	-1 on unsuccessful operation
***		1  on successful operation
**/

int add_noise(projection *p, double percent)
{
    double	*noise;
    double	max = 0;
    int		i;
    int		rnd;

    // Allocating memory for noise
    noise = (double*) malloc(sizeof(double) * p->data_num);

    // Determinig maximum value of projected values
    for ( i = 0; i < p->data_num; i++ )
	if ( max < p->proj_data[i] ) max = p->proj_data[i];

    // Generating given percents of uniformly distributed additive noise
    for ( i = 0; i < p->data_num; i++ ) {
       	rnd = read_random();
	noise[i] = floor(rnd - RAND_MAX / 2.0) * max / RAND_MAX * percent / 50.0;
    }

    // Adding noise to projected values
    for ( i = 0; i < p->data_num; i++ )
	p->proj_data[i] = p->proj_data[i] + noise[i] < 0 ? 0 : p->proj_data[i] + noise[i];

    free(noise);
    return;
}


/**
***
***	get_unif_rand
***
*** get a unform distribution random number between 0 and 1
***
**/
double unif_rand()
{
    return (double)rand()/(double)RAND_MAX;
}

/**
***
***	Mutation
***
**/
/* 
 * number:    the number of existing pictures
 * p:         the vector of existing pictures
 * poss_num:  the possibility of changing number of circles
 * poss_poz:  the possibility of changing pozition of circles
 * poss_size: the possibility of changing size of circles
 */
   
/* itt lehet tbbfajta mutci is: a krk szmnak vltozsa, illetve csak a 
   mretk vltozsa, illetve csak a pozcijuk vltozsa, valamint ezek egytt
   is vltozhatnak */
/* az els kt kr kzppontja nem vltozik */
int mutation(int *number, par_s** p, double poss_num, 
                 double poss_pos, double poss_size)
{
    int i, j, k;
    double *temp;
    par_s *temp_p;
    double x1, x2, y1, y2;
    double max_r, min_r, max_r_temp;
    int new_number;
    new_number = (*number);
    time_t st, en;
    time_t s1, e1, s2, e2;
    double temp_x, temp_y, temp_r;

    
    for (i=0; i<(*number); i++)
    {
        /* Change the numbers of circles*/
        if (unif_rand()<=poss_num)
        {
            temp_p = malloc(sizeof(par_s));
            memcpy(temp_p, p[i], sizeof(par_s));


            if (getrandom(0,1)< 0.5)
            {
time(&s1);
                /* Increase the number of circles */
                temp = malloc(3*(p[i]->circle_num+1)*sizeof(double));
                memcpy(temp, p[i]->params, (3*(p[i]->circle_num)*sizeof(double)));
                temp_p->params = temp;
                if (0 == add_circle_fast(temp_p, 1, min_search_radius))
                {
                    //temp_p->circle_num++;
                    p[new_number] = temp_p;
                    new_number++;
                }
                else
                {
                    if (temp != NULL)
                        free(temp);
                    if (temp_p != NULL)
                        free(temp_p);
                }
time(&e1);
mut1_time += (e1-s1);
            } /* End: Increase */
            else
            {
time(&s1);
                /* Decrease the number of circles */
                if (2 < p[i]->circle_num-1)
                {
                    j = getrandom(2, p[i]->circle_num-1);
                    /* Delete the j-1th circle */
                    temp = malloc(3*(p[i]->circle_num-1)*sizeof(double));
                    temp_p->params = temp;
                    for (k=0; k<j; k++)
                    {
                        temp[k*3]   = p[i]->params[k*3];
                        temp[k*3+1] = p[i]->params[k*3+1];
                        temp[k*3+2] = p[i]->params[k*3+2];
                    }
                    for (k=j; k<p[i]->circle_num-1; k++)
                    {
                        temp[k*3+0] = p[i]->params[k*3+3];
                        temp[k*3+1] = p[i]->params[k*3+4];
                        temp[k*3+2] = p[i]->params[k*3+5];
                    }
                    temp_p->circle_num--;
time(&st);
                    if (cond_check(temp_p)==1)
                    {
                        p[new_number] = temp_p;
                        new_number++;
                    }
                    else
                    {
                        if (temp != NULL)
                            free(temp);
                        if (temp_p != NULL)
                            free(temp_p);
                    }
time(&en);
cond_time += en-st;
                }
time(&e1);
mut2_time += (e1-s1);
            } /* End: Decrease */
        } /* End: Change the number of circles */

time(&s1);
        /* Move circle */
        if (unif_rand()<=poss_pos)
        {
            /* Choose a random circle in the picture */
            if (2 < p[i]->circle_num-1)
            {
                j = getrandom(2, p[i]->circle_num-1);
                /* Choose a random position in the picture in the +/-5 pixel 
                   neighbourhood of the original point */
                x1 = (p[i]->params[3] - p[i]->params[5])<(p[i]->params[3*j]-5) ?
                        (p[i]->params[3*j]-5) : (p[i]->params[3] - p[i]->params[5]);
                y1 = (p[i]->params[4] - p[i]->params[5])<(p[i]->params[3*j+1]-5) ?
                        (p[i]->params[3*j+1]-5) : (p[i]->params[4] - p[i]->params[5]);
                x2 = (p[i]->params[3] + p[i]->params[5])<(p[i]->params[3*j]+5) ?
                        (p[i]->params[3] + p[i]->params[5]) : (p[i]->params[3*j]+5);
                y2 = (p[i]->params[4] + p[i]->params[5])<(p[i]->params[3*j+1]+5) ?
                        (p[i]->params[4] + p[i]->params[5]) : (p[i]->params[3*j+1]+5);
                temp_x   = x1 + unif_rand() * (x2-x1);
                temp_y = y1 + unif_rand() * (y2-y1);
time(&st);
                if (cond_check_fast(p[i], j, temp_x, temp_y, p[i]->params[3*j+2])==1)
                {
                    temp_p = malloc(sizeof(par_s));
                    memcpy(temp_p, p[i], sizeof(par_s));

                    /* Allocate memory for the temp (temporary parameters) */
                    temp = malloc(3*(p[i]->circle_num)*sizeof(double));
                    if (temp == NULL)
                        return -1;
                    temp_p->params = temp;
                    memcpy(temp_p->params, p[i]->params, (3*(p[i]->circle_num)*sizeof(double)));
                    
                    temp_p->params[3*j]   = temp_x;
                    temp_p->params[3*j+1] = temp_y;

                    p[new_number] = temp_p;
                    new_number++;
                }
time(&en);
cond_time += en-st;
            }
        } /* End: Move circle */
time(&e1);
mut3_time += (e1-s1);

        /* Resize circle */
        if (unif_rand()<=poss_size)
        {
time(&s1);
            if (2 < p[i]->circle_num-1)
            {
                /* Choose a random circle in the picture */
                k = getrandom(2, p[i]->circle_num-1);
                min_r = min_search_radius<(p[i]->params[k * 3 + 2]-2.5) ? 
                        (p[i]->params[k * 3 + 2]-2.5) : min_search_radius;
                max_r = p[i]->params[k * 3 + 2]+2.5;
                temp_r = min_r + unif_rand() * (max_r - min_r);
time(&st);
                if (cond_check_fast(p[i], k, p[i]->params[3*k], p[i]->params[3*k+1], temp_r)==1)
                {
                    temp_p = malloc(sizeof(par_s));
                    memcpy(temp_p, p[i], sizeof(par_s));

                    /* Allocate memory for the temp (temporary parameters) */
                    temp = malloc(3*(p[i]->circle_num)*sizeof(double));
                    if (temp == NULL)
                        return -1;
                    temp_p->params = temp;
                    memcpy(temp_p->params, p[i]->params, (3*(p[i]->circle_num)*sizeof(double)));
                    
                    temp_p->params[3*k+2] = temp_r;

                    p[new_number] = temp_p;
                    new_number++;
                }
time(&en);
cond_time += en-st;
            }
time(&e1);
mut4_time += (e1-s1);
        } /* End: Resize circle */
    } /* for */

    *number = new_number;
    return 0;
}


/**
***
***	Cross Over
***
**/

/* ha a random ltal generlt szm kisebb, mint a possibility, akkor az egyed 
   keresztezdik, s a prjt vletlenszeren vlasztjuk ki */
/* az els kt kr csak a sajt pzcijn lev krrel cserldhet */
/* az egyedek szma mindig n, majd a legjobb eredeti kezdpopulci szm egyed
   marad meg a fittness kirtkelse utn */
/* New feature: force cross over - if there is no one offspring after cross over, 
   the algorith gives 50 more chance to the chosen pair */ 
int crossOver(int *number, par_s** p, double possibility){
    int debug_mode = 0;
    par_s *temp_p1;
    par_s *temp_p2;
    double *temp1;
    double *temp2;
    int cut1, cut2;
    int i, j;
    int new_number;
    
    int max_iter = 50; /* The number of iterations if the crossOver isn't successful */
    int iter = 0; /* The current iteration */
//printf("cross ower\n");
    new_number = *number;
    for (i=0; i<(*number); i++)
    {
        if (unif_rand()<possibility)
        {
            while (iter<max_iter)
            {
                /* Choose Pair */
                j=i;
                while (j==i)
                {
                    j = getrandom(0, (*number)-1);
                }
                temp_p1 = malloc(sizeof(par_s));
                memcpy(temp_p1, p[i], sizeof(par_s));
                temp_p2 = malloc(sizeof(par_s));
                memcpy(temp_p2, p[j], sizeof(par_s));
            
                /* crossing over */
                if (temp_p1->circle_num-1 > 2)
                    cut1 = getrandom(2, temp_p1->circle_num-1);
                else
                    cut1 = 2;

                if (temp_p2->circle_num -1 > 2)
                    cut2 = getrandom(2, temp_p2->circle_num-1);
                else
                    cut2 = 2;

                /* The size of the first child is  cut1 + (temp_p2->circle_num - cut2) */
                /* The size of the second child is cut2 + (temp_p1->circle_num - cut1) */
                temp1 = malloc((cut1 + temp_p2->circle_num - cut2)*3*sizeof(double));
                temp_p1->params = temp1;
                memcpy(temp1, p[i]->params, cut1);
                memcpy(temp1+cut1, p[j]->params+cut2, p[j]->circle_num-cut2);

                temp2 = malloc((cut2 + temp_p1->circle_num - cut1)*3*sizeof(double));
                temp_p2->params = temp2;
                memcpy(temp2, p[j]->params, cut2);
                memcpy(temp2+cut2, p[i]->params+cut2, p[i]->circle_num-cut1);


                if (cond_check(temp_p1)==1) {
                   p[new_number] = temp_p1;
                   new_number++;
                   /* the iteration ended, if one new element is successful */
                   iter = max_iter;
                }
                else {
                    free(temp1);
                    free(temp_p1);
                }
                if (cond_check(temp_p2)==1) {
                    p[new_number] = temp_p2;
                    new_number++;
                    /* the iteration ended, if one new element is successful */
                    iter = max_iter;
                }
                else {
                    free(temp2);
                    free(temp_p2);
                }
                iter++;
            } /* while */ 
        } /* if */
    } /* for */
    *number = new_number;
    return 0;
}

/**
***
***	topOfThePops
***
**/

/* Az elemeket gy kellene trlni, hogy csinlunk egy tmbt a sorrendnek, 
   aztn mindig a rendes tmbnk utols elemt betesszk a legrosszabbnak a 
   pozcijra (a legrosszabbat kzben trltk), ha ez a kett megegyezik, 
   akkor az elemet csak smn trljk s az elemek szmt eggyel cskkentjk */
/* Ez rtkeli ki a projekcikat is, majd eltrolja mindegyikhez a 
   clfggvnyrtket, hogy ne kelljen mindig jraszmolni. Ezeket is 
   cserlgetjk. */
/* Teht ha beveszem az elzetes informcit, akkor csak ez vltozik */
int topOfThePops(int *number, int req_number, par_s **p, double *tfunc, 
    double *best_value, int *best_value_index, double *in_proj, init_s *init, 
    int max_circle_number, double* penalty_values, double lambda1, double lambda2
){
    int i, j, l;
    double worst_value;
    projection **projs;
    char* output_name;
    time_t st, en;
    int is_swap;

	/* Allocating memory for projections */
    projs = malloc(init->proj_num * sizeof(projection*));
    for ( i = 0; i < init->proj_num; i++)
        projs[i] = malloc(sizeof(projection));

    for (l=0; l<*number; l++)
    {
        if (tfunc[l] == INF)
        {
            /* Calculate the tfunc value */
            /* Calculate projections */
          	/* Projecting configuration */
time(&st);
/*
            for ( i = 0; i < init->proj_num; i++ ) {
	            proj(projs[i], p[l], init->proj_angle[i], 0, pic_size - 1, (pic_size - 1) / 2.0, (pic_size - 1) / 2.0, 1);
            }
*/
            proj_fast(projs, p[l], init->proj_angle, init->proj_num, 0, pic_size - 1, (pic_size - 1) / 2.0, (pic_size - 1) / 2.0, 1);
time(&en);
proj_time += en-st;
            tfunc[l] = 0;

            if (p[l]->circle_num <= max_circle_number)
            {
                for (i=0; i<init->proj_num; i++)
                {
                    for (j=0; j<projs[i]->data_num; j++)
                    {
                        tfunc[l] += lambda1*abs(projs[i]->proj_data[j] - in_proj[i*pic_size + j]);
                    }
                }
                tfunc[l] += lambda2*penalty_values[p[l]->circle_num-1];
            }
            else
            {
                tfunc[l] = INF;
            }
            
            for (i=0; i < init->proj_num; i++)
            {
                if (projs[i]->proj_data != NULL)
                   free(projs[i]->proj_data);
            }
        }
    }

    for ( i = 0; i < init->proj_num; i++)
    {
        if (projs[i] != NULL)
            free(projs[i]);
    }
    if (projs != NULL)
       free(projs);

    /* Selection: only keep the top of the population */
    i = req_number;
    is_swap = 1;
    while (req_number<*number)
    {
        /* Megkeressk a legrosszabb elemet s kicserljk a kvetkez 
           req_number-nl nagyobb sorszmval, ha az jobb a legrosszabb elemnl */
        if (is_swap)
        {
            l = 0;
            worst_value = tfunc[0];
            for (j=0; j<req_number; j++)
            {
                if (tfunc[j] > worst_value)
                {
                    worst_value = tfunc[j];
                    l = j;
                }
            }
        }
        /* If the i th element better than the l th
           put the i th element to the l th space 
           Delete element from the l th space */
        if (tfunc[l] > tfunc[i])
        {
            if (p[l]->params != NULL)
                free(p[l]->params);
            if (p[l] != NULL)
                free(p[l]);
            /* Move the i th element to the l th space */
            p[l] = p[i];
            tfunc[l] = tfunc[i];
            /* Reset tfunc */
            tfunc[i] = INF;
            is_swap = 1;
//printf("volt csere i: %d  -> l: %d\n", i, l);
        }
        /* else delete the i th element */
        else
        {
            if (p[i]->params != NULL)
                free(p[i]->params);
            if (p[i] != NULL)
                free(p[i]);
            /* Reset tfunc */
            tfunc[i] = INF;
            is_swap = 0;
        }

        i++;
        (*number)--;
    }

    for (l=0; l<*number; l++)
    {
        /* Calculate best_value */
        if (tfunc[l]<*best_value)
        {
            *best_value = tfunc[l];
            *best_value_index = l;
            printf("best_value: %lf\tindex: %d\n", *best_value, *best_value_index);
            
            /* Draw Best Value */
            if (print_best)
            {
                output_name = (char*)malloc(30*sizeof(char));
                sprintf(output_name, "BV%04d.pgm", picture_number++);
                write_PGM(p[*best_value_index], pic_size, pic_size, 255, output_name);
                free(output_name);
            }
        }
    }
//printf("endtop\n");
    return 0;
}

/**
***
***	MAIN
***
**/

/* scriptgener */
#define WORKDIR             "out"
#define INPUT               "filter_o.t"
#define OUTPUT_TRAIN        "c01.data"
#define OUTPUT_TEST         "c01.test"
#define NAMES               "c01.names"
#define FILE_C4_5           "c01"
#define DEFAULT_CONF        "conf.txt"
#define DEFAULT_INPUT_PROJ  "toReconstruct_filter_o.t"
#define DEF_NR_OF_CASE1_TRAIN   100
#define DEF_NR_OF_CASE2_TRAIN   100
#define NR_OF_CIRC1         3
#define NR_OF_CIRC2         4
#define NR_OF_PROJ          4
#define NR_MAX_VALUES       6 /* also defined in local_maxima.c*/

time_t		s_time;
par_s *actualPar;

int main(int argc, char **argv){
    FILE *f2, *f3;
    int *circle;
    int *train_case;
    int nr_circles = 2; /* Default */
    int i, j;
    char temp[200];
    char **argv2;
    par_s **in_par;
    double *tfunc;    /* clfggvny rtkek */
    double *in_proj;  /* input projections */
    int number;
    int req_num;      /* the requested number of the population */
    init_s init;
    int iter = 0;
    int max_iter;         /* the maximum number of iterations */
    double best_value = INF;      /* the minimum value in the target function */
    int best_value_index = 0;     /* the index of the best value in the array */
    double epsilon;       /* if the difference of projections is lower than epsilon, the algorithm stops */
    double cOPoss;         /* possibility of cross over */
    double mut_poss_num;   /* possibility of changeing the number of circles in mutation */
    double mut_poss_pos;   /* possibility of changeing the position of circles in mutation */
    double mut_poss_size;  /* possibility of changeing the size of circles in mutation */
    int max_circle_number;        /* the maximum number of circles */
    double* penalty_values;       /*  */
    double lambda1;
    double lambda2;
time_t mut_time, cross_time, top_time, start_act, end_act;
time_t elapsed_h, elapsed_m, elapsed_s;

    argv2= malloc(20*sizeof(char*));
    for (i=0;i<20; i++)
        argv2[i]= malloc(20*sizeof(char));

    // Using system time to initalize random number generator
    time(&s_time);
    srand(s_time);
    
    i = 1;
    while ( i < argc ) {
        if ( argv[i][0] == '-' ) {
            switch ( argv[i][1] ) {
                default :
                    break;
            }
        } else {
        }
        i++;
    }

    if ( !(f3 = fopen(DEFAULT_CONF, "r")) ) {
         f3 = NULL;
    }
    else {
         fscanf(f3, "%d", &nr_circles);
    }
    circle     = malloc(nr_circles*sizeof(int));
    train_case  = malloc(nr_circles*sizeof(int));

    if (f3 != NULL) {
        for (i=0; i<nr_circles; i++)
            fscanf(f3, "%d", &circle[i]);
        for (i=0; i<nr_circles; i++)
            fscanf(f3, "%d", &train_case[i]);
        fclose(f3);
    }
    else {
        circle[0] = NR_OF_CIRC1;
        circle[1] = NR_OF_CIRC2;
        train_case[0] = DEF_NR_OF_CASE1_TRAIN;
        train_case[1] = DEF_NR_OF_CASE2_TRAIN;
    }
    
    /*Setting the penalty values  */
    max_circle_number = 11; /* Default */
    if ( !(f3 = fopen("penalty.t", "r")) ) {
         f3 = NULL;
    }
    else {
         fscanf(f3, "%d", &max_circle_number);
    }
    
    penalty_values = malloc(max_circle_number*sizeof(double));
    
    if (f3 != NULL) {
        for (i=0; i<max_circle_number; i++)
            fscanf(f3, "%lf", &penalty_values[i]);
        fclose(f3);
    }
    else {
        for (i=0; i<max_circle_number; i++)
        {
            if (i<max_circle_number/2)
            {
                penalty_values[i] = max_circle_number/2 - i;
            }
            else
            {
                penalty_values[i] = i - max_circle_number/2;
            }
        }
    }

    /* Setting the parameters */
    if ( !(f3 = fopen("parameters.t", "r")) ) {
        max_iter = 1000;        /* the maximum number of iterations */
        epsilon = 0.0;          /* if the difference of projections is lower than epsilon, the algorithm stops */
        cOPoss = 0.25;          /* possibility of cross over */
        mut_poss_num  = 0.01;  /* possibility of changeing the number of circles in mutation */
        mut_poss_pos  = 0.01;  /* possibility of changeing the position of circles in mutation */
        mut_poss_size = 0.01;  /* possibility of changeing the size of circles in mutation */
        lambda1 = 100.0 / 40000.0;
        lambda2 = 1.0 / 5.0;
    }
    else {
        fscanf(f3, "%d",  &max_iter);
        fscanf(f3, "%lf", &epsilon);
        fscanf(f3, "%lf", &cOPoss);
        fscanf(f3, "%lf", &mut_poss_num);
        fscanf(f3, "%lf", &mut_poss_pos);
        fscanf(f3, "%lf", &mut_poss_size);
        fscanf(f3, "%lf", &lambda1);
        fscanf(f3, "%lf", &lambda2);
    }
    
    

    sprintf(argv2[0],"projgen");
    sprintf(argv2[1],"-c");
    sprintf(argv2[2],"%d", circle[0]);
    sprintf(argv2[3],"-r");
    sprintf(argv2[4],"-d");
    sprintf(argv2[5],"-q");
    sprintf(argv2[6],"-w");
    sprintf(argv2[7],"%s", WORKDIR);

    for (j=0; j<nr_circles; j++)
    {
        /* generating the initiate population */
        for (i=0; i<train_case[j]; i++)
        {
            sprintf(argv2[2],"%d", circle[j]);
            while(0!=projgen(8, argv2))
            {
                if ( chdir("..") ) {
		            printf("Unable to change working directory.\n");
                    return -1;
                }
            }
            
            if ( chdir("..") ) {
		        printf("Unable to change working directory.\n");
                return -1;
            }
        }
    }

    if ( read_init(INIT_FILE, &init) == -1 ) {
        exit(EXIT_FAILURE);
    }
    
    in_proj = malloc(pic_size*init.proj_num*sizeof(double));
    /* Read input projections to reconstruct */
    if ( !(f3 = fopen(DEFAULT_INPUT_PROJ, "r")) ) {
         printf("Unable to open input projections.\n");
         return -1;
    }
    else {
        for (i=0; i<init.proj_num; i++)
        {
            for (j=0; j<pic_size; j++)
            {
                fscanf(f3, "%lf", &in_proj[i*pic_size + j]);
            }
        }
    }

    /* read data from file */
    if ( chdir(WORKDIR) ) {
        printf("Unable to change working directory.\n");
        return -1;
    }

    /* calculate the maximum possible number of pictures */
    int size = 0;
    for (j=0; j<nr_circles; j++)
    {
        size += train_case[j];
    }

    /* original pictures (1x), crossed over pictures(1x) and mutated pictures(1x) */
    in_par = malloc(8 * size * sizeof(par_s*));
    tfunc  = malloc(8 * size * sizeof(double));
    if ( read_more_params("circles_g.t", in_par, &number) == -1 ) {
        exit(EXIT_FAILURE);
    }

    for (i=0; i<number; i++)
    {
        in_par[i]->lac_cyl = init.lac_cyl;
        in_par[i]->lac_circ = init.lac_circ;
    }
    
    req_num = number;

    for (j=0; j<6*size; j++)
    {
        tfunc[j] = INF;
    }

proj_time = 0;
cond_time = 0;
mut1_time = 0;
mut2_time = 0;
mut3_time = 0;
mut4_time = 0;
cross_time = 0;
mut_time = 0;
top_time = 0;
    /* EVOLUTION */
    while (iter < max_iter && epsilon < best_value){
        /* crossover */
        time(&start_act);
        crossOver(&number, in_par, cOPoss);
        time(&end_act);
cross_time += end_act-start_act;
        /* mutation */
        time(&start_act);
        mutation(&number, in_par, mut_poss_num, mut_poss_pos, mut_poss_size);
        time(&end_act);
mut_time += end_act-start_act;
        /* selection */
        time(&start_act);
        topOfThePops(&number, req_num, in_par, tfunc, &best_value, &best_value_index, 
                     in_proj, &init, max_circle_number, penalty_values, lambda1, lambda2);
        time(&end_act);
top_time += end_act-start_act;

        iter++;
        if (iter % 1000 == 0)
        {
            printf("iter: %d\n", iter);
        }
//printf("iter: %d\tbest_value: %lf\tindex: %d\n", iter, best_value, best_value_index);
    }
    
    elapsed_h = cross_time / 3600;
    elapsed_m = (cross_time - elapsed_h * 3600) / 60;
    elapsed_s = cross_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("cross time: %d sec  --  %02d:%02d:%02d\n", cross_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = mut_time / 3600;
    elapsed_m = (mut_time - elapsed_h * 3600) / 60;
    elapsed_s = mut_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("mut time: %d sec  --  %02d:%02d:%02d\n", mut_time, elapsed_h, elapsed_m, elapsed_s);
    
    elapsed_h = top_time / 3600;
    elapsed_m = (top_time - elapsed_h * 3600) / 60;
    elapsed_s = top_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("top time: %d sec  --  %02d:%02d:%02d\n", top_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = proj_time / 3600;
    elapsed_m = (proj_time - elapsed_h * 3600) / 60;
    elapsed_s = proj_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("proj time: %d sec  --  %02d:%02d:%02d\n", proj_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = cond_time / 3600;
    elapsed_m = (cond_time - elapsed_h * 3600) / 60;
    elapsed_s = cond_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("cond time: %d sec  --  %02d:%02d:%02d\n", cond_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = mut1_time / 3600;
    elapsed_m = (mut1_time - elapsed_h * 3600) / 60;
    elapsed_s = mut1_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("mut1 time: %d sec  --  %02d:%02d:%02d\n", mut1_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = mut2_time / 3600;
    elapsed_m = (mut2_time - elapsed_h * 3600) / 60;
    elapsed_s = mut2_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("mut2 time: %d sec  --  %02d:%02d:%02d\n", mut2_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = mut3_time / 3600;
    elapsed_m = (mut3_time - elapsed_h * 3600) / 60;
    elapsed_s = mut3_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("mut3 time: %d sec  --  %02d:%02d:%02d\n", mut3_time, elapsed_h, elapsed_m, elapsed_s);

    elapsed_h = mut4_time / 3600;
    elapsed_m = (mut4_time - elapsed_h * 3600) / 60;
    elapsed_s = mut4_time - elapsed_h * 3600 - elapsed_m * 60;
    printf("mut4 time: %d sec  --  %02d:%02d:%02d\n", mut4_time, elapsed_h, elapsed_m, elapsed_s);

	/* Saving image of result configuration */
        write_PGM(in_par[best_value_index], pic_size, pic_size, 255, "result.pgm");

    printf("best value: %lf\niter: %d\n", best_value, iter);
    printf("%d\n", in_par[best_value_index]->circle_num);
    for ( i = 0; i < in_par[best_value_index]->circle_num; i++ ) 
	    printf("%g %g %g\n", in_par[best_value_index]->params[3 * i], in_par[best_value_index]->params[3 * i + 1], in_par[best_value_index]->params[3 * i + 2]);


    return 0;
}


/**
***
*** Projgen
***
**/

/* Projection generation */
int projgen(int argc, char **argv)
{
   	par_s		p;
        init_s		init;
        int		circle_num = -1;
        projection	**projs;
        int		i;
        iter_input	i_input;
        FILE		*f;
        time_t		e_time, elapsed_time, elapsed_h, elapsed_m, elapsed_s;
        char		help = FALSE, conf_random = FALSE;
        DIRECT_par	D_par;
        char		fn_par_o[] = IN_PARS_ORIGIN, fn_par_g[] = IN_PARS_GEN;
        char		fn_D_out1[] = DIRECT_ORIGIN, fn_D_out2[] = DIRECT_PROJ;
	char		*work_dir = NULL;


        // Processing command-line
        i = 1;
        while ( i < argc ) {
           if ( argv[i][0] == '-' ) {
              switch ( argv[i][1] ) {
              case '?': help = TRUE;
                 	break;
              case 'c': circle_num = atoi(argv[++i]);
                 	break;
              case 'd': debug = TRUE;
                 	break;
              case 'm': min_search_radius = strtod(argv[++i], NULL);
                 	break;
              case 'q': quiet_mode = TRUE;
                 	break;
              case 'r': conf_random = TRUE;
                 	break;
              case 'w': work_dir = (char *) malloc((strlen(argv[++i]) + 1) * sizeof(char));
	      		strcpy(work_dir, argv[i]);
                 	break;
              default :
                    break;
              }
           } else {
           }
           i++;
        }

	// Changing working directory
	if (work_dir) {
	    if ( chdir(work_dir) ) {
		if ( !quiet_mode ) printf("Unable to change working directory.\nAbnormal termination.\n");
        	exit(EXIT_FAILURE);
	    }
	}

	// Printing help message and exit if required
        if ( help ) {
	    if ( !quiet_mode ) {
        	printf("Usage: projgen [OPTION]...\n\n");
        	printf("Generates projections from random or "
                       "user-specified configurations.\n\n");
		printf("Options:\n");
        	printf("  -?		Display this help and exit.\n");
        	printf("  -c NUM	Use NUM (>=2) circles when "
                       "building random configuration.\n");
        	printf("		Must be specified when using -r.\n");
        	printf("  -d		Enables debug mode.\n");
        	printf("  -m NUM	Minimal radius of allowed circles. "
                       "Default is %g .\n", MIN_SEARCH_RADIUS);
		printf("  -q		Enables quiet mode.\n");
        	printf("  -r		Generate projections from "
                       "random configuration.\n");
	    	printf("  -w DIR	Specifies the working directory.\n");
	    }
            /*exit(EXIT_SUCCESS);*/
            return 0;
        }

        if ( read_init(INIT_FILE, &init) == -1 ) {
            exit(EXIT_FAILURE);
        }

        if ( init.proj_num < 2 ) {
            exit(EXIT_FAILURE);
        }
        
        if ( min_search_radius < EPS ) {
            exit(EXIT_FAILURE);
        }

        p.lac_cyl = init.lac_cyl;
        p.lac_circ = init.lac_circ;

        if ( conf_random ) {
            if ( circle_num == -1 ) {
        	exit(EXIT_FAILURE);
            }
            if ( circle_num < 2 ) {
        	exit(EXIT_FAILURE);
            }
            p.params = malloc(3 * circle_num * sizeof(double));
            add_cylinder(&p);
            if (0!=add_circle(&p, circle_num - 2, min_search_radius))
            {
                return -1;
            }
	    if ( save_params(fn_par_g, &p) == -1 ) {
                exit(EXIT_FAILURE);
	    }
        } else {
	    if ( read_params(fn_par_o, &p) == -1 ) {
                exit(EXIT_FAILURE);
	    }
            if ( cond_check(&p) != 1 ) {
	        free(p.params);
           	exit(EXIT_FAILURE);
	    }
            circle_num = p.circle_num;
        }
        
	// Allocating memory for projections
	i_input.proj_num = init.proj_num;
	i_input.proj_arr = malloc(i_input.proj_num * sizeof(projection*));
        projs = malloc(init.proj_num * sizeof(projection*));
        for ( i = 0; i < init.proj_num; i++) projs[i] = malloc(sizeof(projection));

	// Projecting configuration
        for ( i = 0; i < init.proj_num; i++ ) {
	       proj(projs[i], &p, init.proj_angle[i], 0, pic_size - 1, (pic_size - 1) / 2.0, (pic_size - 1) / 2.0, 1);
               i_input.proj_arr[i] = projs[i];
        }

	// Adding noise
        if (init.use_noise) {
            for (i = 0; i < init.proj_num; i++ ) add_noise(projs[i], init.noise_perc);

        }

	// Saving image of original configuration
/*
    write_PGM(&p, pic_size, pic_size, 255, CONF_ORIGIN);
*/
    // Saving projections
/*
    D_par.i_input = &i_input;
    D_par.in_config = &p;
    D_par.circle_num = circle_num;
    D_par.action = WRDIRECT_IMAGE | WRDIRECT_INPUT;
    if ( conf_random ) {
        D_par.in_fname = fn_par_g;
	} else {
        D_par.in_fname = fn_par_o;
    }
    D_par.fname = fn_D_out1;
    write_DIRECT(&D_par);
    D_par.action ^= WRDIRECT_IMAGE;
    D_par.fname = fn_D_out2;
    i = write_DIRECT(&D_par);

    if ( i == -1 ) {
        exit(EXIT_FAILURE);
    }
*/
    // Freeing allocated memory
    for (i = 0; i < init.proj_num; i++ ) {
        free(projs[i]->proj_data);
        free(projs[i]);
    }
    free(projs);
    free(i_input.proj_arr);
	free(p.params);
	if (work_dir) free(work_dir);

    /*exit(EXIT_SUCCESS);*/
    return 0;
}

