#include			<math.h>
#include			<stdlib.h>
#include			<string.h>
#include			<stdio.h>
#include			<unistd.h>
#include			"specrec.h"



/**
***
***	MACROS & CONSTANTS
***
**/

/**
***
*** this macro generates a random number between the two given integers
***
**/
#define getrandom( min, max ) ((rand() % (int)(((max)+1) - (min))) + (min))

/**
*** Default values of various parameters.
**/
/**
*** Number of projections used.
**/
#define			PROJ_NUM	4
/**
*** Angles of rotations in degrees specifying projecting directions.
**/
#define			PROJ_ANGLE	{0.0, 45.0, 90.0, 135.0}
/**
*** If TRUE, simulated noise will be added to input projections.
**/
#define			USE_NOISE	TRUE
/**
*** Amount of noise in percents to be added to the projections. Let
*** R be a random real number of the interval [0.0; 1.0] (both included),
*** M be the maximal value of a given projection. Then, for every P
*** projection value, the following N is calculated:
***
***    N = (R - 0.5) * M * NOISE_PERC / 50.0 .
***
*** Finally, new projection value P' is calculated as
***
***    P' = MAX(P + N, 0.0),
***
*** where MAX(x, y) returns the greater of the arguments.
***
**/
#define			NOISE_PERC	15
/**
*** LAC (linear attenuation coeffitient) value for cylinder. Should be
*** positive.
**/
#define			LAC_CYL		1.0
/**
*** LAC value for regular circles. Should be positive.
**/
#define			LAC_CIRC	1.0

/**
*** Log file containing output messages.
**/
#define			OUTPUT_LOG	"projgen_o.log"

/**
*** Log file containing error messages.
**/
#define			ERROR_LOG	"projgen_e.log"

/**
*** Name of text file containing original configuration and projections
*** in DIRECT format.
**/
#define			DIRECT_ORIGIN	"origin.txt"

/**
*** Name of initialisation file.
**/
#define			INIT_FILE	"init.t"

/**
*** Name of text file containing parameters for the manually created
*** original configuration.
**/
#define			IN_PARS_ORIGIN	"circles_o.t"

/**
*** Name of text file containing parameters for the randomly generated
*** original configuration.
**/
#define			IN_PARS_GEN	"circles_g.t"

/**
*** Name of text file containing noiseless projections.
**/
#define			FILTER_ORIGIN	"filter_o.t"
/**
*** Name of text file containing noisy projections.
**/
#define			FILTER_NOISY	"filter_n.t"

/**
*** Infinitive
**/
#define          INF             10000000.0

/**
***
***	TYPE DEFINITIONS
***
**/

// Structure for initial projection parameters.
typedef struct {
   			// Number of projections.
                        int		proj_num;
                        // Projecting directions in degrees.
                        double		*proj_angle;
                        // If TRUE, projections will be made noisy.
                        char		use_noise;
                        // Amount of noise in percents to be added.
                        double		noise_perc;
                        // LAC (linear attenuation coeffitient) value for
                        // cylinder. Should be positive.
                        double		lac_cyl;
                        // LAC value for regular circles. Should be positive.
                        double		lac_circ;
} init_s;



/**
***
***	GLOBAL VARIABLES
***
**/

/**
***
***	FUNCTIONS
***
**/

/**
***	Randomly adds cylinder to the given configuration.
***	input:	pointer to a 'par_s' structure containing parameters of the given set of circles
***	output:	--
**/

void add_cylinder(par_s	*in_params)
{
   	int		i;
        int		random;
	double		unif;
        double		outer_start_max = 10.0;
        double		cyl_thick_min = 2.0, cyl_thick_max = 10.0;


        in_params->circle_num = 2;

        // Setting center
        in_params->params[0] = in_params->params[3] = pic_size / 2.0;
        in_params->params[1] = in_params->params[4] = pic_size / 2.0;
        
        // Generating radii
        random = read_random();
        unif = random / (double) RAND_MAX;
        in_params->params[2] = pic_size / 2.0 - unif * outer_start_max;
        random = read_random();
        unif = random / (double) RAND_MAX;
        in_params->params[5] = in_params->params[2] - cyl_thick_min -
                               unif * cyl_thick_max;

	return;
}



/**
***	Reads circle parameters of configuration from file.
***	Input:	name of file
***		address of structure containing parameters on return
***	Output:	1 if operation was successful
***		-1 on unsuccessful operation
**/

int read_params(char	*file_name,
		par_s	*p)
{
    FILE	*f;
    int		circle_num;
    int		i;
    float	x, y, r;

    if ( !(f = fopen(file_name, "r")) ) return -1;

    fscanf(f, "%d\n", &circle_num);
    p->circle_num = circle_num;
    p->params = malloc(3 * circle_num * sizeof(double));
    for ( i = 0; i < circle_num; i++ ) {
	fscanf(f, "%f %f %f\n", &x, &y, &r);
        p->params[3 * i] = x;
	p->params[3 * i + 1] = y;
	p->params[3 * i + 2] = r;
    }

    fclose(f);
    return 1;
}

int read_more_params(char	*file_name,
		par_s	**p, int* number)
{
    FILE	*f;
    int		circle_num;
    int		i, j;
    float	x, y, r;

    if ( !(f = fopen(file_name, "r")) ) return -1;
    j = 0;
    while (fscanf(f, "%d\n", &circle_num)!=EOF)
    {
        
        p[j] = malloc(sizeof(par_s));
        p[j]->circle_num = circle_num;
        p[j]->params = malloc(3 * circle_num * sizeof(double));
        for ( i = 0; i < circle_num; i++ ) {
           	fscanf(f, "%f %f %f\n", &x, &y, &r);
            p[j]->params[3 * i] = x;
            p[j]->params[3 * i + 1] = y;
	        p[j]->params[3 * i + 2] = r;
        }
        j++;
    }
    *number=j;

    fclose(f);
    return 1;
}



/**
***	Reads initial parameters from file, or sets them to default values.
***	Input:	name of file
***		structure containing initalisation parameters on return
***	Output:	-1, if parameter of keyword PROJ_NUM is wrong
***		1  on successful operation
**/

int read_init(char	*file_name,
              init_s	*init)
{
   	FILE		*f;
        char		*buff, *str_par, *str_val;
        double		proj_angle_default[] = PROJ_ANGLE;
        int		i;

    
        if ( !(f = fopen(INIT_FILE, "r")) ) {
           	if (debug) {
                   fprintf(o_log_file, "INIT: Initialisation file not found. Using default values.");
                }
                init->proj_num = PROJ_NUM;
                init->proj_angle = malloc(PROJ_NUM * sizeof(double));
                for (i = 0; i < PROJ_NUM; i++)
                   init->proj_angle[i] = proj_angle_default[i];
                init->use_noise = USE_NOISE;
                init->noise_perc = NOISE_PERC;
                init->lac_cyl = LAC_CYL;
                init->lac_circ = LAC_CIRC;

                return 1;
        } else {
                buff = malloc(256 * sizeof(char));
                str_par = malloc(256 * sizeof(char));
                str_val = malloc(256 * sizeof(char));
                if (debug) {
		   fprintf(o_log_file, "INIT: Reading initialisation values... ");
                }
                init->proj_num = i = 0;
                init->proj_angle = NULL;
                init->use_noise = FALSE;
                init->noise_perc = 0.0;
                init->lac_cyl = LAC_CYL;
                init->lac_circ = LAC_CIRC;
                while ( !feof(f) ) {
                   buff[0] = '\0';
                   fgets(buff, 256, f);
                   buff[255] = '\0';
                   strupcase(buff);
                   str_par[0] = '\0';
                   str_val[0] = '\0';
                   sscanf(buff, " %[A-Za-z_] = %[.0-9eE+-]", str_par, str_val);
                   if ( strlen(str_par) ) {
                      if ( !strcmp("PROJ_NUM", str_par) ) {
                         init->proj_num = atoi(str_val);
                         if ( init->proj_num > 0)
                            init->proj_angle = malloc(init->proj_num * sizeof(double));
                      } else if ( !strcmp("PROJ_ANGLE", str_par) ) {
                         if ( i < init->proj_num )
                            init->proj_angle[i++] = strtod(str_val, NULL);
                      } else if ( !strcmp("NOISE_PERC", str_par) ) {
                         init->use_noise = TRUE;
                         init->noise_perc = strtod (str_val, NULL);
                      } else if ( !strcmp("PIC_SIZE", str_par) ) {
                         pic_size = atoi (str_val);
                      } else if ( !strcmp("LAC_CYL", str_par) ) {
                         init->lac_cyl = strtod (str_val, NULL);
                      } else if ( !strcmp("LAC_CIRC", str_par) ) {
                         init->lac_circ = strtod (str_val, NULL);
                      }
                   }
                }
		fclose(f);
                if (debug) {
		   fprintf(o_log_file, "OK\n");
                }
                free(buff);
                free(str_par);
                free(str_val);

                return i != init->proj_num ? -1 : 1;
        }
}



/**
***	Adds uniformly distributed additive noise to projection.
***	Input:	pointer to a projection to be added noise
***		percentage of added noise
***	Outpu:	-1 on unsuccessful operation
***		1  on successful operation
**/

int add_noise(projection *p, double percent)
{
    double	*noise;
    double	max = 0;
    int		i;
    int		random;

    // Allocating memory for noise
    noise = (double*) malloc(sizeof(double) * p->data_num);

    // Determinig maximum value of projected values
    for ( i = 0; i < p->data_num; i++ )
	if ( max < p->proj_data[i] ) max = p->proj_data[i];

    // Generating given percents of uniformly distributed additive noise
    for ( i = 0; i < p->data_num; i++ ) {
       	random = rand();
	noise[i] = floor(random - RAND_MAX / 2.0) * max / RAND_MAX * percent / 50.0;
    }

    // Adding noise to projected values
    for ( i = 0; i < p->data_num; i++ )
	p->proj_data[i] = p->proj_data[i] + noise[i] < 0 ? 0 : p->proj_data[i] + noise[i];

    free(noise);
    return;
}


/**
***
***	MAIN
***
**/

int main(int argc, char **argv){
    FILE *f, *f2, *f3;
    int i, j;
    par_s **in_par;
    int number;
    int req_num;      /* the requested number of the population */
    init_s init;
    double n_percent = 0.0;
    
    projection **projs;


    if (argc>=2){
        n_percent=atof(argv[1]);
    }

    if ( read_init(INIT_FILE, &init) == -1 ) {
        printf("Something is wrong with the parameter of keyword PROJ_NUM.\n"
	                              "Abnormal termination.\n");
        exit(EXIT_FAILURE);
    }
    
    int size = 1; /* now we only want one picture */

    /* original pictures (1x), crossed over pictures(1x) and mutated pictures(1x) */
    in_par = malloc(6*size * sizeof(par_s*));
    if ( read_more_params("circles_g.t", in_par, &number) == -1 ) {
        printf("Unable to open parameter file.\nAbnormal termination.\n");
		exit(EXIT_FAILURE);
    }

    for (i=0; i<number; i++)
    {
        in_par[i]->lac_cyl = init.lac_cyl;
        in_par[i]->lac_circ = init.lac_circ;
    }
    
	/* Allocating memory for projections */
    projs = malloc(init.proj_num * sizeof(projection*));
    for ( i = 0; i < init.proj_num; i++)
        projs[i] = malloc(sizeof(projection));

	/* Projecting the configuration */
    for ( i = 0; i < init.proj_num; i++ ) {
        proj(projs[i], in_par[0], init.proj_angle[i], 0, pic_size - 1, (pic_size - 1) / 2.0, (pic_size - 1) / 2.0, 1.0);
    }

    if ( f = fopen(FILTER_ORIGIN, "w") ) {
        for (i = 0; i < init.proj_num; i++ ) dump_proj(projs[i], f);
        fclose(f);
    }

    if ( f2 = fopen(FILTER_NOISY, "w") ) {
        for (i = 0; i < init.proj_num; i++ ) {
            add_noise(projs[i], n_percent);
            dump_proj(projs[i], f2);
        }
        fclose(f2);
    }

    /* Saving image of result configuration */
    write_PGM(in_par[0], pic_size, pic_size, 255, "origin.pgm");

    return 0;
}


